#ifndef __CVECTOR3_H
#define __CVECTOR3_H
////////////////////////////////////////////////////////////////////////////////
#include <math.h>
////////////////////////////////////////////////////////////////////////////////
class cVector2;
class cVector4;

class cVector3 {
public:
  // these usions allow the cVector3 to be used as a color component
  union {
    float d_x, d_r;
  };
  union {
    float d_y, d_g;
  };
  union {
    float d_z, d_b;
  };

  ////////////////////////////////////////////////////////////////////////////////
  // constructors & destructors
  ////////////////////////////////////////////////////////////////////////////////
  cVector3() {  // Null constructor
    d_x = 0;
    d_y = 0;
    d_z = 0;
  }

  
  cVector3( float x, float y, float z ) {
    d_x = x;
    d_y = y;
    d_z = z;
  }
  
  
  cVector3( const cVector3 &v ) {
    d_x = v.d_x;
    d_y = v.d_y;
    d_z = v.d_z;
  }
  
  
  cVector3( float v[ 3 ] ) {
    d_x = v[ 0 ];
    d_y = v[ 1 ];
    d_z = v[ 2 ];
  }

  
  ~cVector3() {};


  ////////////////////////////////////////////////////////////////////////////////
  // operators
  ////////////////////////////////////////////////////////////////////////////////

  inline cVector3 operator = ( const cVector3 &v ) {  // assignment
    d_x = v.d_x;
    d_y = v.d_y;
    d_z = v.d_z;

    return *this;
  }


  inline cVector3 operator - () {  // Negation
    d_x = -d_x; 
    d_y = -d_y; 
    d_z = -d_z;

    return *this;
  }

  inline cVector3 operator *= ( float v ) {  // assigned multiply by a float
    d_x *= v;
    d_y *= v;
    d_z *= v;

    return *this;
  }

  inline cVector3 operator /= ( float t ) {  // assigned division by a float
    float v;

    if( !t ) v = 0;
    else v = 1.0f / t;

    d_x *= v;
    d_y *= v;
    d_z *= v;

    return *this;
  }

  inline cVector3 operator -= ( const cVector3 &v ) {  // assigned subtraction
    d_x -= v.d_x;
    d_y -= v.d_y;
    d_z -= v.d_z;

    return *this;
  }

  inline cVector3 operator += ( const cVector3 &v ) {  // assigned addition
    d_x += v.d_x;
    d_y += v.d_y;
    d_z += v.d_z;

    return *this;
  }

  inline cVector3 operator ^= ( const cVector3 &v ) {  // assigned cross product
    float nx, ny, nz;
    
    nx = ( d_y * v.d_z - d_z * v.d_y );
    ny =-( d_x * v.d_z - d_z * v.d_x );
    nz = ( d_x * v.d_y - d_y * v.d_x );
    d_x = nx;
    d_y = ny;
    d_z = nz;

    return *this;
  }

  inline int operator == ( const cVector3 &v ) {
    if( fabs( d_x - v.d_x ) < 0.01f &&
        fabs( d_y - v.d_y ) < 0.01f &&
        fabs( d_z - v.d_z ) < 0.01f ) return 1;  // match
    return 0;  // no match
  }

  inline int operator != ( const cVector3 &v ) {
    if( fabs( d_x - v.d_x ) < 0.01f &&
        fabs( d_y - v.d_y ) < 0.01f &&
        fabs( d_z - v.d_z ) < 0.01f ) return 0;  // no match
    return 1;  // match
  }


// METHODS
  inline float Length() {
    return (float)sqrt( *this | *this );
  }

  inline float LengthSquared() {
    return *this | *this;
  }

  inline void Normalize() {
    float len, iLen;

    len = Length();
    if( !len ) iLen = 0;
    else iLen = 1.0f / len;

    d_x *= iLen;
    d_y *= iLen;
    d_z *= iLen;
  }

  inline void ClampMin( float min ) {  // Clamp to minimum
    if( d_x < min ) d_x = min;
    if( d_y < min ) d_y = min;
    if( d_z < min ) d_z = min;
  }

  inline void ClampMax( float max ) {  // Clamp to maximum
    if( d_x > max ) d_x = max;
    if( d_y > max ) d_y = max;
    if( d_z > max ) d_z = max;
  }

  inline void Clamp( float min, float max ) {  // Clamp to range ]min,max[
    ClampMin( min );
    ClampMax( max );
  }

  inline void Interpolate( const cVector3 &v, float a ) {  // Interpolate between *this and v
    float b = 1.0f - a;

    d_x = b * d_x + a * v.d_x;
    d_y = b * d_y + a * v.d_y;
    d_z = b * d_z + a * v.d_z;
  }

  inline void Position( float x, float y, float z ) {
    d_x = x;
    d_y = y;
    d_z = z;
  }


  inline void RotateAboutLine( const cVector3 &line, float angle ) {
    cVector3  rotate[ 3 ], vec;
    float     S, C, iC;

    vec.d_x = line.d_x;
    vec.d_y = line.d_y;
    vec.d_z = line.d_z;

    // Calculate some constants beforehand
    S = (float)sin( angle );
    C = (float)cos( angle );
    iC = 1.0f - C;

    // Create the rotation matrix for rotation We could have used cMatrix4
    // but then it causes even more interdependence, not a good thing.
    rotate[ 0 ].d_x = C + iC * vec.d_x * vec.d_x;
    rotate[ 0 ].d_y =     iC * vec.d_x * vec.d_y + S * vec.d_z;
    rotate[ 0 ].d_z =     iC * vec.d_x * vec.d_z - S * vec.d_y;
    rotate[ 1 ].d_x =     iC * vec.d_y * vec.d_x - S * vec.d_z;
    rotate[ 1 ].d_y = C + iC * vec.d_y * vec.d_y;
    rotate[ 1 ].d_z =     iC * vec.d_y * vec.d_z + S * vec.d_x;
    rotate[ 2 ].d_x =     iC * vec.d_z * vec.d_x + S * vec.d_y;
    rotate[ 2 ].d_y =     iC * vec.d_z * vec.d_y - S * vec.d_x;
    rotate[ 2 ].d_z = C + iC * vec.d_z * vec.d_z;

    // Save the original coordinates before we modify them
    vec.d_x = d_x;
    vec.d_y = d_y;
    vec.d_z = d_z;

    // Apply the rotation
    d_x = vec | rotate[ 0 ];
    d_y = vec | rotate[ 1 ];
    d_z = vec | rotate[ 2 ];
  }


  inline float operator | ( const cVector3 &v ) {  // Dot product
    return d_x * v.d_x + d_y * v.d_y + d_z * v.d_z;
  }


  inline cVector3 cVector3::operator / ( float t ) {  // vector / float
    if( !t ) return cVector3( 0, 0, 0 );

    float s = 1.0f / t;

    return cVector3( d_x * s, d_y * s, d_z * s );
  }


  inline cVector3 cVector3::operator + ( const cVector3 &b ) {  // vector + vector
    return cVector3( d_x + b.d_x, d_y + b.d_y, d_z + b.d_z );
  }


  inline cVector3 cVector3::operator - ( const cVector3 &b ) {  // vector - vector
    return cVector3( d_x - b.d_x, d_y - b.d_y, d_z - b.d_z );
  }


  inline cVector3 cVector3::operator ^ ( const cVector3 &b ) {  // cross(a,b)
    float nx, ny, nz;

    nx =  d_y * b.d_z - d_z * b.d_y;
    ny =  d_z * b.d_x - d_x * b.d_z;
    nz =  d_x * b.d_y - d_y * b.d_x;

    return cVector3( nx, ny, nz );
  }


  inline cVector3 cVector3::operator * ( float s ) {
    return cVector3( d_x * s, d_y * s, d_z * s );
  }


  // This starts to make things ugly...
  cVector3 operator = ( const cVector4 &v );
};


#endif

