#ifndef __CMATRIX4_H
#define __CMATRIX4_H
////////////////////////////////////////////////////////////////////////////////
#ifndef __CVECTOR3_H
#include "cVector3.h"
#endif
#ifndef __CVECTOR4_H
#include "cVector4.h"
#endif
////////////////////////////////////////////////////////////////////////////////
class cMatrix4 {
public:
  float   a00, a01, a02, a03,
          a10, a11, a12, a13,
          a20, a21, a22, a23,
          a30, a31, a32, a33;
  char    d_GSCount;
  ////////////////////////////////////////////////////////////////////////////////
  // constructors & destructors
  ////////////////////////////////////////////////////////////////////////////////

  cMatrix4() {
    MakeIdentity();
  }

  
  cMatrix4( float v[ 16 ] ) {
    a00 = v[ 0  ];  a01 = v[ 1  ];  a02 = v[ 2  ];  a03 = v[ 3  ];
    a10 = v[ 4  ];  a11 = v[ 5  ];  a12 = v[ 6  ];  a13 = v[ 7  ];
    a20 = v[ 8  ];  a21 = v[ 9  ];  a22 = v[ 10 ];  a23 = v[ 11 ];
    a30 = v[ 12 ];  a31 = v[ 13 ];  a32 = v[ 14 ];  a33 = v[ 15 ];
  }


  ~cMatrix4() {}
  ////////////////////////////////////////////////////////////////////////////////
  // methods
  ////////////////////////////////////////////////////////////////////////////////
  void MakeNull(){
    a00 = a01 = a02 = a03 = 0.0f;
    a10 = a11 = a12 = a13 = 0.0f;
    a20 = a21 = a22 = a23 = 0.0f;
    a30 = a31 = a32 = a33 = 0.0f;
    d_GSCount = 0;
  }


  void MakeIdentity() {
    MakeNull();
    a00 = a11 = a22 = a33 = 1.0f;
    d_GSCount = 0;
  }


  void Invert() {
    float x, y, z;

    x   = a01;  y   = a02;  z   = a03;
    a01 = a10;  a02 = a20;  a03 = a30;
    a10 =   x;  a20 =   y;  a30 =   z;
    x   = a12;  y   = a13;  z   = a23;
    a12 = a21;  a13 = a31;  a23 = a32;
    a21 =   x;  a31 =   y;  a32 =   z;
  }
  
  
  // re-orthogonalize matrix
  inline void GrahmShmidt() {
    cVector3 a, b, c;
    float    dot;

    a.d_x = a00;  a.d_y = a01;  a.d_z = a02;
    b.d_x = a10;  b.d_y = a11;  b.d_z = a12;
    c.d_x = a20;  c.d_y = a21;  c.d_z = a22;
    a.Normalize();
    dot = b | a;
    b -= a * dot;
    b.Normalize();
    c = b ^ a;

    // return the values to the matrix
    a00 = a.d_x;  a01 = a.d_y;  a02 = a.d_z;
    a10 = b.d_x;  a11 = b.d_y;  a12 = b.d_z;
    a20 = c.d_x;  a21 = c.d_y;  a22 = c.d_z;
    d_GSCount = 0;
  }


  inline void RotateAroundLocalAxis( int axisId, float angle ) {
    if( axisId < 0 || axisId > 2 || !angle ) return;

    cVector3 a, b, c;

    // a - axis we turn around
    switch( axisId ) {
      case 0:
        a.Position( a00, a01, a02 );
        b.Position( a10, a11, a12 );
        c.Position( a20, a21, a22 );
        break;
      case 1:
        b.Position( a00, a01, a02 );
        a.Position( a10, a11, a12 );
        c.Position( a20, a21, a22 );
        break;
      case 2:
        b.Position( a00, a01, a02 );
        c.Position( a10, a11, a12 );
        a.Position( a20, a21, a22 );
        break;
    }

    b.RotateAboutLine( a, angle );
    c.RotateAboutLine( a, angle );

    switch( axisId ) {
      case 0:
        a00 = a.d_x;  a01 = a.d_y;  a02 = a.d_z;
        a10 = b.d_x;  a11 = b.d_y;  a12 = b.d_z;
        a20 = c.d_x;  a21 = c.d_y;  a22 = c.d_z;
        break;
      case 1:
        a00 = b.d_x;  a01 = b.d_y;  a02 = b.d_z;
        a10 = a.d_x;  a11 = a.d_y;  a12 = a.d_z;
        a20 = c.d_x;  a21 = c.d_y;  a22 = c.d_z;
        break;
      case 2:
        a00 = b.d_x;  a01 = b.d_y;  a02 = b.d_z;
        a10 = c.d_x;  a11 = c.d_y;  a12 = c.d_z;
        a20 = a.d_x;  a21 = a.d_y;  a22 = a.d_z;
        break;
    }

    d_GSCount++;
    if( d_GSCount > 32 ) GrahmShmidt();
  }


  inline cMatrix4 operator =( const cMatrix4 &src ) {
    a00 = src.a00;
    a01 = src.a01;
    a02 = src.a02;
    a03 = src.a03;
    a10 = src.a10;
    a11 = src.a11;
    a12 = src.a12;
    a13 = src.a13;
    a20 = src.a20;
    a21 = src.a21;
    a22 = src.a22;
    a23 = src.a23;
    a30 = src.a30;
    a31 = src.a31;
    a32 = src.a32;
    a33 = src.a33;

    return *this;
  }


  // add two matrixes
  inline cMatrix4 operator +( const cMatrix4 &src ) {
    float pVList[ 16 ];

    pVList[ 0 ] = a00 + src.a00;
    pVList[ 1 ] = a01 + src.a01;
    pVList[ 2 ] = a02 + src.a02;
    pVList[ 3 ] = a03 + src.a03;

    pVList[ 4 ] = a10 + src.a10;
    pVList[ 5 ] = a11 + src.a11;
    pVList[ 6 ] = a12 + src.a12;
    pVList[ 7 ] = a13 + src.a13;

    pVList[ 8 ] = a20 + src.a20;
    pVList[ 9 ] = a21 + src.a21;
    pVList[ 10 ] = a22 + src.a22;
    pVList[ 11 ] = a23 + src.a23;

    pVList[ 12 ] = a30 + src.a30;
    pVList[ 13 ] = a31 + src.a31;
    pVList[ 14 ] = a32 + src.a32;
    pVList[ 15 ] = a33 + src.a33;

    return cMatrix4( pVList );
  }
  
  
  // concatenate two matrixes
  inline cMatrix4 operator *( const cMatrix4 &src ) {
    float pVList[ 16 ];

    pVList[ 0 ] = a00 * src.a00 + a01 * src.a10 + a02 * src.a20 + a03 * src.a30;
    pVList[ 1 ] = a00 * src.a01 + a01 * src.a11 + a02 * src.a21 + a03 * src.a31;
    pVList[ 2 ] = a00 * src.a02 + a01 * src.a12 + a02 * src.a22 + a03 * src.a32;
    pVList[ 3 ] = a00 * src.a03 + a01 * src.a13 + a02 * src.a23 + a03 * src.a33;

    pVList[ 4 ] = a10 * src.a00 + a11 * src.a10 + a12 * src.a20 + a13 * src.a30;
    pVList[ 5 ] = a10 * src.a01 + a11 * src.a11 + a12 * src.a21 + a13 * src.a31;
    pVList[ 6 ] = a10 * src.a02 + a11 * src.a12 + a12 * src.a22 + a13 * src.a32;
    pVList[ 7 ] = a10 * src.a03 + a11 * src.a13 + a12 * src.a23 + a13 * src.a33;

    pVList[ 8 ] = a20 * src.a00 + a21 * src.a10 + a22 * src.a20 + a23 * src.a30;
    pVList[ 9 ] = a20 * src.a01 + a21 * src.a11 + a22 * src.a21 + a23 * src.a31;
    pVList[ 10 ] = a20 * src.a02 + a21 * src.a12 + a22 * src.a22 + a23 * src.a32;
    pVList[ 11 ] = a20 * src.a03 + a21 * src.a13 + a22 * src.a23 + a23 * src.a33;

    pVList[ 12 ] = a30 * src.a00 + a31 * src.a10 + a32 * src.a20 + a33 * src.a30;
    pVList[ 13 ] = a30 * src.a01 + a31 * src.a11 + a32 * src.a21 + a33 * src.a31;
    pVList[ 14 ] = a30 * src.a02 + a31 * src.a12 + a32 * src.a22 + a33 * src.a32;
    pVList[ 15 ] = a30 * src.a03 + a31 * src.a13 + a32 * src.a23 + a33 * src.a33;

    d_GSCount++;
    if( d_GSCount > 32 ) GrahmShmidt();

    return cMatrix4( pVList );
  }


  friend cVector4 operator *( const cVector3 &src, const cMatrix4 &trans );
  friend cVector4 operator *( const cVector4 &src, const cMatrix4 &trans );
};


#endif

