/* Project: DirectX Windows DLL
 * File   : cDInput.cpp
 * Created: 24/05/98 Dan Royer
 * Note   : Can this be done better?
 */
#ifndef __CDINPUT_H
#include "cDInput.h"
#endif
////////////////////////////////////////////////////////////////////////////////
// Why do these have to be defined when they already exist in dinput.h?!?
static const GUID   GUID_SysKeyboard = { 0x6F1D2B61,0xD5A0,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00 };
static const GUID   GUID_Key         = { 0x55728220,0xD33C,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00 };

static const GUID   GUID_SysMouse    = { 0x6F1D2B60,0xD5A0,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00 };
static const GUID   GUID_XAxis       = { 0xA36D02E0,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00 };
static const GUID   GUID_YAxis       = { 0xA36D02E1,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00 };
static const GUID   GUID_ZAxis       = { 0xA36D02E2,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00 };
/*
DEFINE_GUID(GUID_RxAxis,  0xA36D02F4,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00);
DEFINE_GUID(GUID_RyAxis,  0xA36D02F5,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00);
DEFINE_GUID(GUID_RzAxis,  0xA36D02E3,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00);
DEFINE_GUID(GUID_Slider,  0xA36D02E4,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00);

DEFINE_GUID(GUID_Button,  0xA36D02F0,0xC9F3,0x11CF,0xBF,0xC7,0x44,0x45,0x53,0x54,0x00,0x00);
*/

static cKeyId gKeyNames[] = {
  { "ESC", DI_INPUTKEYBD, 0x01, 0, 0 },
  { "1", DI_INPUTKEYBD, 0x02, '1', '!' },
  { "2", DI_INPUTKEYBD, 0x03, '2', '@' },
  { "3", DI_INPUTKEYBD, 0x04, '3', '#' },
  { "4", DI_INPUTKEYBD, 0x05, '4', '$' },
  { "5", DI_INPUTKEYBD, 0x06, '5', '%' },
  { "6", DI_INPUTKEYBD, 0x07, '6', '^' },
  { "7", DI_INPUTKEYBD, 0x08, '7', '&' },
  { "8", DI_INPUTKEYBD, 0x09, '8', '*' },
  { "9", DI_INPUTKEYBD, 0x0A, '9', '(' },
  { "0", DI_INPUTKEYBD, 0x0B, '0', ')' },
  { "-", DI_INPUTKEYBD, 0x0C, '-', '_' },
  { "=", DI_INPUTKEYBD, 0x0D, '=', '+' },
  { "BACKSPACE", DI_INPUTKEYBD, 0x0E, 1, 1 },
  { "TAB", DI_INPUTKEYBD, 0x0F, 0, 0 },

  { "Q", DI_INPUTKEYBD, 0x10, 'q', 'Q' },
  { "W", DI_INPUTKEYBD, 0x11, 'w', 'W' },
  { "E", DI_INPUTKEYBD, 0x12, 'e', 'E' },
  { "R", DI_INPUTKEYBD, 0x13, 'r', 'R' },
  { "T", DI_INPUTKEYBD, 0x14, 't', 'T' },
  { "Y", DI_INPUTKEYBD, 0x15, 'y', 'Y' },
  { "U", DI_INPUTKEYBD, 0x16, 'u', 'U' },
  { "I", DI_INPUTKEYBD, 0x17, 'i', 'I' },
  { "O", DI_INPUTKEYBD, 0x18, 'o', 'O' },
  { "P", DI_INPUTKEYBD, 0x19, 'p', 'P' },
  { "[", DI_INPUTKEYBD, 0x1A, '[', '{' },
  { "]", DI_INPUTKEYBD, 0x1B, ']', '}' },
  { "RETURN", DI_INPUTKEYBD, 0x1C, 0, 0 },
  { "LEFT CTRL", DI_INPUTKEYBD, 0x1D, 0, 0 },

  { "A", DI_INPUTKEYBD, 0x1E, 'a', 'A' },
  { "S", DI_INPUTKEYBD, 0x1F, 's', 'S' },
  { "D", DI_INPUTKEYBD, 0x20, 'd', 'D' },
  { "F", DI_INPUTKEYBD, 0x21, 'f', 'F' },
  { "G", DI_INPUTKEYBD, 0x22, 'g', 'G' },
  { "H", DI_INPUTKEYBD, 0x23, 'h', 'H' },
  { "J", DI_INPUTKEYBD, 0x24, 'j', 'J' },
  { "K", DI_INPUTKEYBD, 0x25, 'k', 'K' },
  { "L", DI_INPUTKEYBD, 0x26, 'l', 'L' },
  { ";", DI_INPUTKEYBD, 0x27, ';', ':' },
  { "'", DI_INPUTKEYBD, 0x28, '\'', '"' },
  { "`", DI_INPUTKEYBD, 0x29, '`', '~' },
  { "LEFT SHIFT", DI_INPUTKEYBD, 0x2A, 0, 0 },
  { "\\", DI_INPUTKEYBD, 0x2B, '\\', '|' },

  { "Z", DI_INPUTKEYBD, 0x2C, 'z', 'Z' },
  { "X", DI_INPUTKEYBD, 0x2D, 'x', 'X' },
  { "C", DI_INPUTKEYBD, 0x2E, 'c', 'C' },
  { "V", DI_INPUTKEYBD, 0x2F, 'v', 'V' },
  { "B", DI_INPUTKEYBD, 0x30, 'b', 'B' },
  { "N", DI_INPUTKEYBD, 0x31, 'n', 'N' },
  { "M", DI_INPUTKEYBD, 0x32, 'm', 'M' },
  { ",", DI_INPUTKEYBD, 0x33, ',', '<' },
  { ".", DI_INPUTKEYBD, 0x34, '.', '<' },
  { "/", DI_INPUTKEYBD, 0x35, '/', '?' },
  { "RIGHT SHIFT", DI_INPUTKEYBD, 0x36, 0, 0 },
  { "*", DI_INPUTKEYBD, 0x37, '*', '*' },
  { "LEFT MENU", DI_INPUTKEYBD, 0x38, 0, 0 },
  { "SPACE", DI_INPUTKEYBD, 0x39, ' ', ' ' },
  { "CAPSLOCK", DI_INPUTKEYBD, 0x3A, 0, 0 },

  { "F1", DI_INPUTKEYBD, 0x3B, 0, 0 },
  { "F2", DI_INPUTKEYBD, 0x3C, 0, 0 },
  { "F3", DI_INPUTKEYBD, 0x3D, 0, 0 },
  { "F4", DI_INPUTKEYBD, 0x3E, 0, 0 },
  { "F5", DI_INPUTKEYBD, 0x3F, 0, 0 },
  { "F6", DI_INPUTKEYBD, 0x40, 0, 0 },
  { "F7", DI_INPUTKEYBD, 0x41, 0, 0 },
  { "F8", DI_INPUTKEYBD, 0x42, 0, 0 },
  { "F9", DI_INPUTKEYBD, 0x43, 0, 0 },
  { "F10", DI_INPUTKEYBD, 0x44, 0, 0 },
  { "NUMLOCK", DI_INPUTKEYBD, 0x45, 0, 0 },
  { "SCROLLLOCK", DI_INPUTKEYBD, 0x46, 0, 0 },

  { "NUMPAD7", DI_INPUTKEYBD, 0x47, '7', '7' },
  { "NUMPAD8", DI_INPUTKEYBD, 0x48, '8', '8' },
  { "NUMPAD9", DI_INPUTKEYBD, 0x49, '9', '9' },
  { "NUMPAD-", DI_INPUTKEYBD, 0x4A, '-', '-' },
  { "NUMPAD4", DI_INPUTKEYBD, 0x4B, '4', '4' },
  { "NUMPAD5", DI_INPUTKEYBD, 0x4C, '5', '5' },
  { "NUMPAD6", DI_INPUTKEYBD, 0x4D, '6', '6' },
  { "NUMPAD+", DI_INPUTKEYBD, 0x4E, '+', '+' },
  { "NUMPAD1", DI_INPUTKEYBD, 0x4F, '1', '1' },
  { "NUMPAD2", DI_INPUTKEYBD, 0x50, '2', '2' },
  { "NUMPAD3", DI_INPUTKEYBD, 0x51, '3', '3' },
  { "NUMPAD0", DI_INPUTKEYBD, 0x52, '0', '0' },
  { "NUMPAD.", DI_INPUTKEYBD, 0x53, '.', '.' },

  { "F11", DI_INPUTKEYBD, 0x57, 0, 0 },
  { "F12", DI_INPUTKEYBD, 0x58, 0, 0 },
  { "F13", DI_INPUTKEYBD, 0x64, 0, 0 },
  { "F14", DI_INPUTKEYBD, 0x65, 0, 0 },
  { "F15", DI_INPUTKEYBD, 0x66, 0, 0 },

  { "KANA", DI_INPUTKEYBD, 0x70, 0, 0 },
  { "CONVERT", DI_INPUTKEYBD, 0x79, 0, 0 },
  { "NOCONVERT", DI_INPUTKEYBD, 0x7B, 0, 0 },
  { "YEN", DI_INPUTKEYBD, 0x7D, 0, 0 },
  { "NUMPAD=", DI_INPUTKEYBD, 0x8D, '=', '=' },
  { "CIRCUMFLEX", DI_INPUTKEYBD, 0x90, 0, 0 },
  { "AT", DI_INPUTKEYBD, 0x91, 0, 0 },
  { ":", DI_INPUTKEYBD, 0x92, 0, 0 },
  { "UNDERLINE", DI_INPUTKEYBD, 0x93, 0, 0 },
  { "KANJI", DI_INPUTKEYBD, 0x94, 0, 0 },
  { "STOP", DI_INPUTKEYBD, 0x95, 0, 0 },
  { "AX", DI_INPUTKEYBD, 0x96, 0, 0 },
  { "UNLABELED", DI_INPUTKEYBD, 0x97, 0, 0 },

  { "ENTER", DI_INPUTKEYBD, 0x9C, 0, 0 },
  { "RIGHT CTRL", DI_INPUTKEYBD, 0x9D, 0, 0 },
  { "NUMPAD,", DI_INPUTKEYBD, 0xB3, 0, 0 },
  { "NUMPAD//", DI_INPUTKEYBD, 0xB5, '/', '/' },
  { "SYSRQ", DI_INPUTKEYBD, 0xB7, 0, 0 },
  { "RIGHT MENU", DI_INPUTKEYBD, 0xB8, 0, 0 },
  { "HOME", DI_INPUTKEYBD, 0xC7, 0, 0 },
  { "UP", DI_INPUTKEYBD, 0xC8, 0, 0 },
  { "PRIOR", DI_INPUTKEYBD, 0xC9, 0, 0 },
  { "LEFT", DI_INPUTKEYBD, 0xCB, 1, 1 },
  { "RIGHT", DI_INPUTKEYBD, 0xCD, 0, 0 },
  { "END", DI_INPUTKEYBD, 0xCF, 0, 0 },
  { "DOWN", DI_INPUTKEYBD, 0xD0, 0, 0 },
  { "NEXT", DI_INPUTKEYBD, 0xD1, 0, 0 },
  { "INSERT", DI_INPUTKEYBD, 0xD2, 0, 0 },
  { "DELETE", DI_INPUTKEYBD, 0xD3, 1, 1 },
  { "LEFT WIN", DI_INPUTKEYBD, 0xDB, 0, 0 },
  { "RIGHT WIN", DI_INPUTKEYBD, 0xDC, 0, 0 },
  { "APPS", DI_INPUTKEYBD, 0xDD, 0, 0 },

  { "MBUTTON1", DI_INPUTMOUSE, 0, 0, 0 },
  { "MBUTTON2", DI_INPUTMOUSE, 1, 0, 0 },
  { "MBUTTON3", DI_INPUTMOUSE, 2, 0, 0 },
  { "MBUTTON4", DI_INPUTMOUSE, 3, 0, 0 },
  { "MROLLUP", DI_INPUTMOUSE, 4, 0, 0 },
  { "MROLLDOWN", DI_INPUTMOUSE, 5, 0, 0 },

  { "JSLIDER", DI_INPUTJOYST, 32, 0, 0 },
  { "JPOV", DI_INPUTJOYST, 33, 0, 0 },
  { "JBUTTON1", DI_INPUTJOYST, 1, 0, 0 },
  { "JBUTTON2", DI_INPUTJOYST, 2, 0, 0 },
  { "JBUTTON3", DI_INPUTJOYST, 3, 0, 0 },
  { "JBUTTON4", DI_INPUTJOYST, 4, 0, 0 },
  { "JBUTTON5", DI_INPUTJOYST, 5, 0, 0 },
  { "JBUTTON6", DI_INPUTJOYST, 6, 0, 0 },
  { "JBUTTON7", DI_INPUTJOYST, 7, 0, 0 },
  { "JBUTTON8", DI_INPUTJOYST, 8, 0, 0 },
  { "JBUTTON9", DI_INPUTJOYST, 9, 0, 0 },
  { "JBUTTON10", DI_INPUTJOYST, 10, 0, 0 },
  { "JBUTTON11", DI_INPUTJOYST, 11, 0, 0 },
  { "JBUTTON12", DI_INPUTJOYST, 12, 0, 0 },
  { "JBUTTON13", DI_INPUTJOYST, 13, 0, 0 },
  { "JBUTTON14", DI_INPUTJOYST, 14, 0, 0 },
  { "JBUTTON15", DI_INPUTJOYST, 15, 0, 0 },
  { "JBUTTON16", DI_INPUTJOYST, 16, 0, 0 },
  { "JBUTTON17", DI_INPUTJOYST, 17, 0, 0 },
  { "JBUTTON18", DI_INPUTJOYST, 18, 0, 0 },
  { "JBUTTON19", DI_INPUTJOYST, 19, 0, 0 },
  { "JBUTTON20", DI_INPUTJOYST, 20, 0, 0 },
  { "JBUTTON21", DI_INPUTJOYST, 21, 0, 0 },
  { "JBUTTON22", DI_INPUTJOYST, 22, 0, 0 },
  { "JBUTTON23", DI_INPUTJOYST, 23, 0, 0 },
  { "JBUTTON24", DI_INPUTJOYST, 24, 0, 0 },
  { "JBUTTON25", DI_INPUTJOYST, 25, 0, 0 },
  { "JBUTTON26", DI_INPUTJOYST, 26, 0, 0 },
  { "JBUTTON27", DI_INPUTJOYST, 27, 0, 0 },
  { "JBUTTON28", DI_INPUTJOYST, 28, 0, 0 },
  { "JBUTTON29", DI_INPUTJOYST, 29, 0, 0 },
  { "JBUTTON30", DI_INPUTJOYST, 30, 0, 0 },
  { "JBUTTON31", DI_INPUTJOYST, 31, 0, 0 },
  { "JBUTTON32", DI_INPUTJOYST, 32, 0, 0 },
};
////////////////////////////////////////////////////////////////////////////////

cDInput::cDInput() {
  d_hinst = NULL;
  d_hwnd = NULL;
  d_di = NULL;
  d_DIKeyboard = NULL;
  d_DIMouse = NULL;
  d_keybdAquired = 0;
  d_mouseAquired = 0;
  d_keybdInitialized = 0;
  d_keybdInitialized = 0;

  d_mouseScale = 1.0f;
}


cDInput::~cDInput() {
  KeyboardRelease();
  MouseRelease();

  if( d_di ) d_di->Release();
  d_di = NULL;

  UnBindAll();
}


int cDInput::Init( HINSTANCE hInst, HWND hwnd ) {
  d_hinst = hInst;
  d_hwnd = hwnd;

  // try to create DirectInput object (DIRECTINPUT_VERSION == DX5)
  if( DirectInputCreate( d_hinst, DIRECTINPUT_VERSION, &d_di, NULL ) != DI_OK ) {
    // creation failed, try DX3 compatibility
    if( DirectInputCreate( d_hinst, 0x0300, &d_di, NULL ) != DI_OK ) return ZVDI_INITFAILURE;
  }

  return ZVDI_OK;
}


int cDInput::UpdateAll() {
  cAction *pAction;

  for( pAction = d_actions.GetNext(); pAction; pAction = pAction->GetNext() ) {
    switch( pAction->d_state ) {
      case ACTION_OFF:  break;  // off
      case ACTION_BEGUN:  pAction->d_state = ACTION_ON;  break;  // starting
      case ACTION_ON:  break;  // on
      case ACTION_ENDED:  pAction->d_state = ACTION_OFF;  break;  // ending
    }
  }

  if( KeyboardUpdate() ) return ZVDI_UPDATEFAILURE;
  if( MouseUpdate() ) return ZVDI_UPDATEFAILURE;

  return ZVDI_OK;
}


// try to acquire the keyboard
int cDInput::ReaquireAll() {
  if( KeyboardReaquire() ) return ZVDI_ACQUIREFAILURE;
  if( MouseReaquire() ) return ZVDI_ACQUIREFAILURE;

  return ZVDI_OK;
}


void cDInput::ReleaseAll() {
  KeyboardRelease();
  MouseRelease();
}


int cDInput::KeyboardInit() {
  KeyboardRelease();

  // Create keyboard device
  if( d_di->CreateDevice( GUID_SysKeyboard, &d_DIKeyboard, NULL ) != DI_OK ) return ZVDI_INITFAILURE;

  // Tell DirectInput that we want to receive data in keyboard format
  if( d_DIKeyboard->SetDataFormat( &c_dfDIKeyboard ) != DI_OK ) return ZVDI_INITFAILURE;

  // set cooperative level
  if( d_DIKeyboard->SetCooperativeLevel( d_hwnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND ) != DI_OK ) return ZVDI_INITFAILURE;

  d_keybdInitialized = 1;

  if( KeyboardReaquire() ) return ZVDI_ACQUIREFAILURE;

  return ZVDI_OK;
}


int cDInput::KeyboardUpdate() {
  int       i, amnt, shift;
  cAction   *pAction;
  HRESULT   hRes;

  if( !d_keybdInitialized ) return 0;

  memcpy( d_keybdData[ 1 ], d_keybdData[ 0 ], 256 );

  hRes = d_DIKeyboard->GetDeviceState( sizeof( d_keybdData[ 0 ] ), d_keybdData[ 0 ] );
  if( hRes != DI_OK && hRes == DIERR_INPUTLOST ) {
    if( KeyboardReaquire() ) return ZVDI_ACQUIREFAILURE;
  }

  /* Update the d_keySequence buffer.  It does NOT
   * compensate for the backspace key and that ilk,
   * it should be done somewhere else.
   */

  // is shift being held down?
  shift = 0;
  amnt = sizeof( gKeyNames ) / sizeof( cKeyId );
  for( i = 0; i < amnt; i++ ) {
    if( strcmp( "LEFT SHIFT", gKeyNames[ i ].d_name ) &&
        strcmp( "RIGHT SHIFT", gKeyNames[ i ].d_name ) ) continue;
    if( d_keybdData[ 0 ][ gKeyNames[ i ].d_id ] ) {
      shift = 1;
      break;
    }
  }

  for( i = 0; i < 256; i++ ) {
    int k, j;

    // only interpret new keys
    if( !d_keybdData[ 0 ][ i ] || d_keybdData[ 1 ][ i ] ) continue;
    // locate said key in the gKeyNames table
    amnt = sizeof( gKeyNames ) / sizeof( cKeyId );
    for( j = 0; j < amnt; j++ ) {
      if( gKeyNames[ j ].d_id == i ) break;
    }
    if( j != amnt ) {
      if( shift ) k = gKeyNames[ j ].d_asciiShift;
      else k = gKeyNames[ j ].d_ascii;
    } else k = 0;

    // don't overrun the buffer!
    if( strlen( d_keySequence ) >= 99 ) {
      for( j = 0; j < 99; j++ ) d_keySequence[ j ] = d_keySequence[ j + 1 ];
      d_keySequence[ 99 ] = 0;
    }
    // insert the ASCII key code, if any.
    if( k != 1 ) d_keySequence[ strlen( d_keySequence ) ] = k;
    else if( strlen( d_keySequence ) ) {
      d_keySequence[ strlen( d_keySequence ) - 1 ] = 0;  // backspace, delete, etc...
    }
  }

  // Update the actions
  for( pAction = d_actions.GetNext(); pAction; pAction = pAction->GetNext() ) {
    if( pAction->d_pKey->d_inputType != DI_INPUTKEYBD ) continue;
    i = d_keybdData[ 0 ][ pAction->d_pKey->d_id ];
    if( !i && !( pAction->d_state & ACTION_ON ) ) continue;
    if( i && ( pAction->d_state & ACTION_ON ) ) continue;
    if( pAction->d_state == ACTION_OFF ) pAction->d_state = ACTION_BEGUN;
    else if( pAction->d_state == ACTION_ON ) pAction->d_state = ACTION_ENDED;
    else continue;
    pAction->d_start = GetTickCount();
  }

  return ZVDI_OK;
}


// try to acquire the keyboard
int cDInput::KeyboardReaquire() {
  HRESULT err;

  if( !d_keybdInitialized || !d_DIKeyboard ) return 0;

  d_keybdAquired = 0;
  err = d_DIKeyboard->Acquire();
  if( err == DI_OK || err == S_FALSE ) d_keybdAquired = 1;  // acquired again?

  if( !d_keybdAquired ) return ZVDI_ACQUIREFAILURE;

  memset( d_keySequence, 0, sizeof( d_keySequence ) );  // clear the buffer

  return ZVDI_OK;
}


void cDInput::KeyboardRelease() {
  if( d_keybdAquired ) {
    d_DIKeyboard->Unacquire();
    d_DIKeyboard->Release();
  }
  d_keybdInitialized = 0;
  d_keybdAquired = 0;
  d_DIKeyboard = NULL;
}
////////////////////////////////////////////////////////////////////////////////

// initialize the mouse
int cDInput::MouseInit( int exclusive ) {  
  MouseRelease();

  // Create mouse device
  if( d_di->CreateDevice( GUID_SysMouse, &d_DIMouse, NULL ) != DI_OK ) return ZVDI_INITFAILURE;

  // Tell DirectInput that we want to receive data in mouse format
  if( d_DIMouse->SetDataFormat( &c_dfDIMouse ) != DI_OK ) return ZVDI_INITFAILURE;

  // Set cooperative level
  if( exclusive ) {
    if( d_DIMouse->SetCooperativeLevel( d_hwnd, DISCL_EXCLUSIVE | DISCL_FOREGROUND ) != DI_OK ) return ZVDI_INITFAILURE;
  } else {
    if( d_DIMouse->SetCooperativeLevel( d_hwnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND ) != DI_OK ) return ZVDI_INITFAILURE;
  }
/*
  // This came from the Scribble tutorial.  Hmm...mouse thread...hmm...
  d_mouseHandle = CreateEvent( FALSE, FALSE, FALSE, FALSE );
  if( !d_mouseHandle ) return 4;

  if( d_DIMouse->SetEventNotification( d_mouseHandle ) != DI_OK ) return 5;
*/
  d_mouseInitialized = 1;

  if( MouseReaquire() ) return ZVDI_ACQUIREFAILURE;  // try to acquire the mouse

  return ZVDI_OK;
}


int cDInput::MouseUpdate() {
  cAction       *pAction;
  DIMOUSESTATE  dims;
  HRESULT       hRes;
  int           data;

  if( !d_mouseInitialized ) return 0;

  hRes = DIERR_INPUTLOST;
  // if input is lost then acquire and keep trying 
  while( DIERR_INPUTLOST == hRes ) {
    // get the input's device state, and put the state in dims
    hRes = d_DIMouse->GetDeviceState( sizeof( DIMOUSESTATE ), &dims );

    if( hRes == DIERR_INPUTLOST && MouseReaquire() ) return ZVDI_ACQUIREFAILURE;
  }

  d_mouseData[ 0 ] = dims.lX;
  d_mouseData[ 1 ] = dims.lY;
  if( dims.lZ < 0 ) d_mouseData[ 2 ] = -1;
  else if( dims.lZ > 0 ) d_mouseData[ 2 ] = 1;
  else d_mouseData[ 2 ] = 0;
  d_mouseData[ 3 ] = ( dims.rgbButtons[ 0 ] != 0 );
  d_mouseData[ 4 ] = ( dims.rgbButtons[ 1 ] != 0 );
  d_mouseData[ 5 ] = ( dims.rgbButtons[ 2 ] != 0 );
  d_mouseData[ 6 ] = ( dims.rgbButtons[ 3 ] != 0 );

  for( pAction = d_actions.GetNext(); pAction; pAction = pAction->GetNext() ) {
    if( pAction->d_pKey->d_inputType != DI_INPUTMOUSE ) continue;
    data = d_mouseData[ 3 + pAction->d_pKey->d_id ];
    if( !data && !( pAction->d_state & ACTION_ON ) ) continue;
    if( data && ( pAction->d_state & ACTION_ON ) ) continue;
    if( pAction->d_state == ACTION_OFF ) pAction->d_state = ACTION_BEGUN;
    else if( pAction->d_state == ACTION_ON ) pAction->d_state = ACTION_ENDED;
    else continue;
    pAction->d_start = GetTickCount();
  }

  // Bad C++ - this is really inefficient.  Speed it up?
  for( pAction = d_actions.GetNext(); pAction; pAction = pAction->GetNext() ) {
    if( pAction->d_pKey->d_inputType != DI_INPUTMOUSE ) continue;
    if( !strcmp( pAction->d_pKey->d_name, "MROLLUP" ) ) {
      if( ( d_mouseData[ 2 ] != 1 && !( pAction->d_state & ACTION_ON ) ) ||
          ( d_mouseData[ 2 ] == 1 && ( pAction->d_state & ACTION_ON ) ) ) continue;
    } else if( !strcmp( pAction->d_pKey->d_name, "MROLLDOWN" ) ) {
      if( ( d_mouseData[ 2 ] != -1 && !( pAction->d_state & ACTION_ON ) ) ||
          ( d_mouseData[ 2 ] == -1 && ( pAction->d_state & ACTION_ON ) ) ) continue;
    } else continue;
    if( pAction->d_state == ACTION_OFF ) pAction->d_state = ACTION_BEGUN;
    else if( pAction->d_state == ACTION_ON ) pAction->d_state = ACTION_ENDED;
    else continue;
    pAction->d_start = GetTickCount();
  }


  return ZVDI_OK;
}


int cDInput::MouseReaquire() {
  HRESULT err;

  if( !d_mouseInitialized || !d_DIMouse ) return 0;

  d_mouseAquired = 0;
  err = d_DIMouse->Acquire();
  if( err == DI_OK || err == S_FALSE ) d_mouseAquired = 1;

  if( !d_mouseAquired ) return ZVDI_ACQUIREFAILURE;

  return ZVDI_OK;
}


void cDInput::MouseRelease() {
  if( d_mouseAquired ) {
    d_DIMouse->Unacquire();
    d_DIMouse->Release();
  }
  d_mouseInitialized = 0;
  d_mouseAquired = 0;
  d_DIMouse = NULL;
}
////////////////////////////////////////////////////////////////////////////////

int cDInput::BindKeyToAction( char *pKeyStr, char *pActionStr ) {
  cAction *pNewAction, *pNextAction;
  int i, amnt;

  if( !pKeyStr || !pActionStr ) return ZVDI_INVALIDPARAM;

  // check key name is valid
  amnt = sizeof( gKeyNames ) / sizeof( cKeyId );
  for( i = 0; i < amnt; i++ ) {
    if( !stricmp( gKeyNames[ i ].d_name, pKeyStr ) ) break;
  }
  if( i == amnt ) return ZVDI_INVALIDPARAM;

  // prevent the same key from being bound to the same action
  for( pNewAction = d_actions.GetNext(); pNewAction; pNewAction = pNextAction ) {
    pNextAction = pNewAction->GetNext();
    if( !stricmp( pNewAction->d_name, pActionStr ) &&
        !stricmp( pNewAction->d_pKey->d_name, pKeyStr ) ) break;
  }
  if( pNewAction ) return ZVDI_OK;  // exists already

  pNewAction = new cAction();
  strcpy( pNewAction->d_name, pActionStr );
  pNewAction->d_pKey = &gKeyNames[ i ];
  pNewAction->d_start = GetTickCount();

  pNextAction = d_actions.GetNext();
  pNewAction->SetNext( pNextAction );
  pNewAction->SetPrev( &d_actions );
  if( pNextAction ) pNextAction->SetPrev( pNewAction );
  d_actions.SetNext( pNewAction );

  return ZVDI_OK;
}


int cDInput::UnBindKeyFromAction( char *pKeyStr, char *pActionStr ) {
  cAction *pAction, *pNextAction;

  if( !pKeyStr || !pActionStr ) return ZVDI_INVALIDPARAM;

  for( pAction = d_actions.GetNext(); pAction; pAction = pNextAction ) {
    pNextAction = pAction->GetNext();
    if( stricmp( pAction->d_name, pActionStr ) ) continue;
    if( stricmp( pAction->d_pKey->d_name, pKeyStr ) ) continue;
    delete pAction;
  }

  return ZVDI_OK;
}


int cDInput::UnBindKeyFromAll( char *pKeyStr ) {
  cAction *pAction, *pNextAction;

  if( !pKeyStr ) return ZVDI_INVALIDPARAM;

  for( pAction = d_actions.GetNext(); pAction; pAction = pNextAction ) {
    pNextAction = pAction->GetNext();
    if( stricmp( pAction->d_pKey->d_name, pKeyStr ) ) continue;
    delete pAction;
  }

  return ZVDI_OK;
}


int cDInput::UnBindAllFromAction( char *pActionStr ) {
  cAction *pAction, *pNextAction;

  if( !pActionStr ) return ZVDI_INVALIDPARAM;

  for( pAction = d_actions.GetNext(); pAction; pAction = pNextAction ) {
    pNextAction = pAction->GetNext();
    if( stricmp( pAction->d_name, pActionStr ) ) continue;
    delete pAction;
  }

  return ZVDI_OK;
}


int cDInput::UnBindAll() {
  cAction *pAction, *pNextAction;

  for( pAction = d_actions.GetNext(); pAction; pAction = pNextAction ) {
    pNextAction = pAction->GetNext();
    delete pAction;
  }

  return ZVDI_OK;
}


int cDInput::QueryAction( cActionQuery *pQuery ) {
  DWORD         onTime, offTime;
  cAction       *pAction;
  int           exists;
  eActionState  state;

  if( !pQuery ) return ZVDI_INVALIDPARAM;

  onTime = GetTickCount();
  offTime = 0;
  exists = 0;
  state = ACTION_OFF;
  pQuery->d_state = ACTION_OFF;

  for( pAction = d_actions.GetNext(); pAction; pAction = pAction->GetNext() ) {
    if( stricmp( pAction->d_name, pQuery->d_name ) ) continue;
    exists = 1;

    switch( pAction->d_state ) {
      case ACTION_ON:
        state = ACTION_ON;
        if( onTime > pAction->d_start ) onTime = pAction->d_start;
        break;
      case ACTION_BEGUN:
        if( state == ACTION_ON ) continue;
        state = ACTION_BEGUN;
        if( onTime > pAction->d_start ) onTime = pAction->d_start;
        break;
      case ACTION_ENDED:
        if( state & ACTION_ON ) continue;
        if( state == ACTION_OFF ) state = ACTION_ENDED;
        if( offTime < pAction->d_start ) offTime = pAction->d_start;
        break;
      case ACTION_OFF:
        if( state != ACTION_OFF ) continue;
        if( offTime < pAction->d_start ) offTime = pAction->d_start;
        break;
    }
  }
  if( !exists ) return ZVDI_ACTIONNONEXISTENT;

  if( state & ACTION_ON ) pQuery->d_start = onTime;
  else pQuery->d_start = offTime;
  pQuery->d_state = state;

  return ZVDI_OK;
}


int cDInput::QueryInputDevice( cActionQuery *pQuery ) {
  if( !pQuery ) return ZVDI_INVALIDPARAM;

  eActionState state;

  state = ACTION_OFF;
  pQuery->d_state = ACTION_OFF;

  int amnt, i;

  amnt = sizeof( gKeyNames ) / sizeof( cKeyId );
  for( i = 0; i < amnt; i++ ) {
    if( !stricmp( gKeyNames[ i ].d_name, pQuery->d_name ) ) break;
  }
  if( i == amnt ) return ZVDI_ACTIONNONEXISTENT;

  if( d_keybdData[ 0 ][ gKeyNames[ i ].d_id ] ) {
    if( d_keybdData[ 1 ][ gKeyNames[ i ].d_id ] ) state = ACTION_ON;
    else state = ACTION_BEGUN;
  } else {
    if( d_keybdData[ 1 ][ gKeyNames[ i ].d_id ] ) state = ACTION_ENDED;
    else state = ACTION_OFF;
  }

  pQuery->d_start = GetTickCount();
  pQuery->d_state = state;

  return ZVDI_OK;
}

