/* NETWORK CLIENT: cNetworkClient.cpp
 * Created 22/06/99 Dan Royer
 * The main class for the application
 */
#if !defined( __CNETWORKCLIENT_H )
#include "cNetworkClient.h"
#endif
#if !defined( __CPATCH_H )
#include "cPatch.h"
#endif
#if !defined( __CMSG_H )
#include "cMsg.h"
#endif
#include <commctrl.h>
#include "resource.h"
////////////////////////////////////////////////////////////////////////////////

cNetworkClient::cNetworkClient() : cWindow() {
  d_isActive = 0;
  d_numPatches = 4;
  d_isReady = 0;
/*
  d_patches[ 0 ].d_controlPoints[ 0 ].Position( -240, -240,   0 );
  d_patches[ 0 ].d_controlPoints[ 1 ].Position(    0, -460,   0 );
  d_patches[ 0 ].d_controlPoints[ 2 ].Position(  240, -240,   0 );
  d_patches[ 0 ].d_controlPoints[ 3 ].Position(  -80,  -80,   0 );
  d_patches[ 0 ].d_controlPoints[ 4 ].Position(    0, -150,   0 );
  d_patches[ 0 ].d_controlPoints[ 5 ].Position(   80,  -80,   0 );
  d_patches[ 0 ].d_controlPoints[ 6 ].Position(  -80,  -80, 240 );
  d_patches[ 0 ].d_controlPoints[ 7 ].Position(    0, -150, 240 );
  d_patches[ 0 ].d_controlPoints[ 8 ].Position(   80,  -80, 240 );

  d_patches[ 1 ].d_controlPoints[ 0 ].Position(  240,  240,   0 );
  d_patches[ 1 ].d_controlPoints[ 1 ].Position(    0,  460,   0 );
  d_patches[ 1 ].d_controlPoints[ 2 ].Position( -240,  240,   0 );
  d_patches[ 1 ].d_controlPoints[ 3 ].Position(   80,   80,   0 );
  d_patches[ 1 ].d_controlPoints[ 4 ].Position(    0,  150,   0 );
  d_patches[ 1 ].d_controlPoints[ 5 ].Position(  -80,   80,   0 );
  d_patches[ 1 ].d_controlPoints[ 6 ].Position(   80,   80, 240 );
  d_patches[ 1 ].d_controlPoints[ 7 ].Position(    0,  150, 240 );
  d_patches[ 1 ].d_controlPoints[ 8 ].Position(  -80,   80, 240 );

  d_patches[ 2 ].d_controlPoints[ 0 ].Position(  240, -240,   0 );
  d_patches[ 2 ].d_controlPoints[ 1 ].Position(  460,    0,   0 );
  d_patches[ 2 ].d_controlPoints[ 2 ].Position(  240,  240,   0 );
  d_patches[ 2 ].d_controlPoints[ 3 ].Position(   80,  -80,   0 );
  d_patches[ 2 ].d_controlPoints[ 4 ].Position(  150,    0,   0 );
  d_patches[ 2 ].d_controlPoints[ 5 ].Position(   80,   80,   0 );
  d_patches[ 2 ].d_controlPoints[ 6 ].Position(   80,  -80, 240 );
  d_patches[ 2 ].d_controlPoints[ 7 ].Position(  150,    0, 240 );
  d_patches[ 2 ].d_controlPoints[ 8 ].Position(   80,   80, 240 );

  d_patches[ 3 ].d_controlPoints[ 0 ].Position( -240,  240,   0 );
  d_patches[ 3 ].d_controlPoints[ 1 ].Position( -460,    0,   0 );
  d_patches[ 3 ].d_controlPoints[ 2 ].Position( -240, -240,   0 );
  d_patches[ 3 ].d_controlPoints[ 3 ].Position(  -80,   80,   0 );
  d_patches[ 3 ].d_controlPoints[ 4 ].Position( -150,    0,   0 );
  d_patches[ 3 ].d_controlPoints[ 5 ].Position(  -80,  -80,   0 );
  d_patches[ 3 ].d_controlPoints[ 6 ].Position(  -80,   80, 240 );
  d_patches[ 3 ].d_controlPoints[ 7 ].Position( -150,    0, 240 );
  d_patches[ 3 ].d_controlPoints[ 8 ].Position(  -80,  -80, 240 );
*/
  d_patches[ 0 ].d_controlPoints[ 0 ].Position( -240, -240,   0 );
  d_patches[ 0 ].d_controlPoints[ 1 ].Position( -120, -240,   0 );
  d_patches[ 0 ].d_controlPoints[ 2 ].Position(    0, -240,   0 );
  d_patches[ 0 ].d_controlPoints[ 3 ].Position( -240, -120,   0 );
  d_patches[ 0 ].d_controlPoints[ 4 ].Position( -120, -120,   0 );
  d_patches[ 0 ].d_controlPoints[ 5 ].Position(    0, -120,   0 );
  d_patches[ 0 ].d_controlPoints[ 6 ].Position( -240,    0,   0 );
  d_patches[ 0 ].d_controlPoints[ 7 ].Position( -120,    0,   0 );
  d_patches[ 0 ].d_controlPoints[ 8 ].Position(    0,    0,   0 );
                                                              
  d_patches[ 1 ].d_controlPoints[ 0 ].Position( -240,    0,   0 );
  d_patches[ 1 ].d_controlPoints[ 1 ].Position( -120,    0,   0 );
  d_patches[ 1 ].d_controlPoints[ 2 ].Position(    0,    0,   0 );
  d_patches[ 1 ].d_controlPoints[ 3 ].Position( -240,  120,   0 );
  d_patches[ 1 ].d_controlPoints[ 4 ].Position( -120,  120,   0 );
  d_patches[ 1 ].d_controlPoints[ 5 ].Position(    0,  120,   0 );
  d_patches[ 1 ].d_controlPoints[ 6 ].Position( -240,  240,   0 );
  d_patches[ 1 ].d_controlPoints[ 7 ].Position( -120,  240,   0 );
  d_patches[ 1 ].d_controlPoints[ 8 ].Position(    0,  240,   0 );
                                                              
  d_patches[ 2 ].d_controlPoints[ 0 ].Position(    0, -240,   0 );
  d_patches[ 2 ].d_controlPoints[ 1 ].Position(  120, -240,   0 );
  d_patches[ 2 ].d_controlPoints[ 2 ].Position(  240, -240,   0 );
  d_patches[ 2 ].d_controlPoints[ 3 ].Position(    0, -120,   0 );
  d_patches[ 2 ].d_controlPoints[ 4 ].Position(  120, -120,   0 );
  d_patches[ 2 ].d_controlPoints[ 5 ].Position(  240, -120,   0 );
  d_patches[ 2 ].d_controlPoints[ 6 ].Position(    0,    0,   0 );
  d_patches[ 2 ].d_controlPoints[ 7 ].Position(  120,    0,   0 );
  d_patches[ 2 ].d_controlPoints[ 8 ].Position(  240,    0,   0 );
                                                              
  d_patches[ 3 ].d_controlPoints[ 0 ].Position(    0,    0,   0 );
  d_patches[ 3 ].d_controlPoints[ 1 ].Position(  120,    0,   0 );
  d_patches[ 3 ].d_controlPoints[ 2 ].Position(  240,    0,   0 );
  d_patches[ 3 ].d_controlPoints[ 3 ].Position(    0,  120,   0 );
  d_patches[ 3 ].d_controlPoints[ 4 ].Position(  120,  120,   0 );
  d_patches[ 3 ].d_controlPoints[ 5 ].Position(  240,  120,   0 );
  d_patches[ 3 ].d_controlPoints[ 6 ].Position(    0,  240,   0 );
  d_patches[ 3 ].d_controlPoints[ 7 ].Position(  120,  240,   0 );
  d_patches[ 3 ].d_controlPoints[ 8 ].Position(  240,  240,   0 );
}


cNetworkClient::~cNetworkClient() {
  Shutdown();
}
////////////////////////////////////////////////////////////////////////////////

int cNetworkClient::Startup( HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow ) {
  INITCOMMONCONTROLSEX icce;
  MSG                  msg;

  // d_hInstance is declared in cWindow
  d_hInstance = hInst;

  icce.dwSize = sizeof( icce );
  icce.dwICC = ICC_INTERNET_CLASSES;
  InitCommonControlsEx( &icce );

  if( d_options.Init( hInst, IDD_CONNECTDIALOG ) ) return 1;
  if( d_options.Create() ) return 1;
  d_options.Show( 1 );

  while( GetMessage( &msg, NULL, 0, 0 ) ) {
    TranslateMessage( &msg );
    DispatchMessage( &msg );
  }

  if( d_options.GetCancel() ) return 1;

  // Create a window
  if( cWindow::Init( d_hInstance, "NetClient" ) ) {
    MessageBox( NULL, "Window creation failed.", "Error!", MB_APPLMODAL );
    return 1;
  }
  if( cWindow::Register() ) {
    MessageBox( NULL, "Window creation failed.", "Error!", MB_APPLMODAL );
    return 1;
  }
  if( cWindow::Create( "Network Client" ) ) {
    MessageBox( NULL, "Window creation failed.", "Error!", MB_APPLMODAL );
    return 1;
  }

  // Set up DirectDraw
#if defined( _DEBUG )
  if( d_D2D.Init( 640, 580, 16, 0, d_hwnd ) ) {
    cWindow::Destroy();
    MessageBox( NULL, "Input capture failed.", "Error!", MB_APPLMODAL );
    return 1;
  }
#else
  if( d_D2D.Init( 640, 580, 16, 1, d_hwnd ) ) {
    cWindow::Destroy();
    MessageBox( NULL, "Input capture failed.", "Error!", MB_APPLMODAL );
    return 1;
  }
#endif

  // Show the window
  cWindow::Show( 1 );
  cWindow::Update();
  d_isReady = 1;

  // Capture keyboard & mouse with DirectInput
  if( d_dinput.Init( d_hInstance, d_hwnd ) || d_dinput.KeyboardInit() || d_dinput.MouseInit() ) {
    cWindow::Destroy();
    MessageBox( NULL, "Input capture failed.", "Error!", MB_APPLMODAL );
    return 1;
  }

  // Bind keys?
  d_dinput.BindKeyToAction( "LEFT", "turnLeft" );
  d_dinput.BindKeyToAction( "numpad4", "turnLeft" );
  d_dinput.BindKeyToAction( "RIGHT", "turnRight" );
  d_dinput.BindKeyToAction( "numpad6", "turnRight" );
  d_dinput.BindKeyToAction( "UP", "accel+" );
  d_dinput.BindKeyToAction( "numpad8", "accel+" );
  d_dinput.BindKeyToAction( "DOWN", "accel-" );
  d_dinput.BindKeyToAction( "numpad2", "accel-" );
  d_dinput.BindKeyToAction( "F", "Weapon+" );  // not used
  d_dinput.BindKeyToAction( "S", "Weapon-" );  // not used
  d_dinput.BindKeyToAction( "SPACE", "firePrimary" );

  d_dinput.BindKeyToAction( "F6", "LOD-" );
  d_dinput.BindKeyToAction( "F7", "LOD+" );
  d_dinput.BindKeyToAction( "ESC", "quit" );
  d_dinput.BindKeyToAction( "`", "consoleToggle" );

  // connect to the server
  d_clientThread.Startup( d_hInstance, NULL, d_hwnd, &d_log );
  d_clientThread.SetName( d_options.GetLogin() );
  d_clientThread.SetPass( d_options.GetPass() );

  char buffer[ 64 ];
  sprintf( buffer, "%d.%d.%d.%d",
    d_options.GetIP( 0 ), d_options.GetIP( 1 ),
    d_options.GetIP( 2 ), d_options.GetIP( 3 ) );

  d_console.Init( &d_D2D, &d_log );
  d_console.SetState( 1 );
  d_log.Log( "Network Client\nCreated by Dan Royer circa 07/1999\naggravated@bigfoot.com\n" );
  d_log.Log( "--------------------------------\nConnecting to %s:%d...\n", buffer, d_options.GetPort() );

  d_clientThread.GetHost( buffer, d_options.GetPort() );
  // when OnGetHost is called we'll start trying to connect to the server.

  return 0;
}


int cNetworkClient::MainLoop() {
  MSG msg;

  d_lastTime = GetTickCount();

  for( ;; ) {
    if( d_isActive ) {  //  When window is not minimized
      if( PeekMessage( &msg, NULL, 0, 0, PM_NOREMOVE ) ) {  // Any windows messages?
        if( !GetMessage( &msg, NULL, 0, 0 ) ) break;
        TranslateMessage( &msg );
        DispatchMessage( &msg );
      } else if( Update() ) break;  // Update client positions
    } else {  // When minimized GetMessage returns 0
      if( !GetMessage( &msg, NULL, 0, 0 ) ) break;
      TranslateMessage( &msg );
      DispatchMessage( &msg );
    }
  }
  Shutdown();

  return 0;
}


void cNetworkClient::Shutdown() {
  // Disconnect from server
  d_clientThread.Shutdown();

  // Release DirectInput keyboard & mouse
  d_dinput.ReleaseAll();

  // Cleanup DirectDraw
  d_D2D.DestroyObjects();

  // Destroy window.
  cWindow::Destroy();
}


int cNetworkClient::Update() {
  DWORD currTime, deltaTime;

  // Determine deltatime
  currTime = GetTickCount();
  deltaTime = currTime - d_lastTime;
  if( !deltaTime ) deltaTime = 1;

  // Read & process network data
  d_clientThread.Update( deltaTime );

  // Don't continue if the window isn't ready
  if( !d_isReady ) return 0;

  cMatrix4                        cameraMatrix, finalMatrix;
  tcLinkedList<cPlayer>           *pLocalClient;
  tcLinkedListItterator<cMissile> itterMissile;
  tcLinkedListItterator<cPlayer>  itterClient;
  cVector3                        vel, accel;
  cActionQuery                    query;
  int                             i;

  // Read and process keyboard data
  d_dinput.UpdateAll();

  // Game config & etc... commands
  strcpy( query.d_name, "consoleToggle" );
  d_dinput.QueryAction( &query );
  if( query.d_state == ACTION_BEGUN ) {
    memset( d_dinput.d_keySequence, 0, sizeof( d_dinput.d_keySequence ) );
    d_console.SetState( d_console.GetState() ^ 1 );
  }

  strcpy( query.d_name, "quit" );
  d_dinput.QueryAction( &query );
  if( query.d_state == ACTION_ON ) return 1;

  strcpy( query.d_name, "LOD+" );
  d_dinput.QueryAction( &query );
  if( query.d_state == ACTION_ENDED ) {
    for( i = 0; i < d_numPatches; i++ ) d_patches[ i ].LOD( -1 );
  }
  strcpy( query.d_name, "LOD-" );
  d_dinput.QueryAction( &query );
  if( query.d_state == ACTION_ENDED ) {
    for( i = 0; i < d_numPatches; i++ ) d_patches[ i ].LOD( 1 );
  }

  // Intialize data for motion
  pLocalClient = d_clientThread.GetClients();

  if( pLocalClient->GetSecurity() ) {  // must be logged in first
    if( pLocalClient->GetState() == STATE_DEAD ) {
      // If player is dead and logged in, spawn when the fire key is hit
      strcpy( query.d_name, "firePrimary" );
      d_dinput.QueryAction( &query );
      if( query.d_state == ACTION_ENDED ) {
        // Bad C++ - shots should be EVENTS.  This is very bad.
        pLocalClient->SetMotionFlags( pLocalClient->GetMotionFlags() | MOTION_FIRESTART );
      }
    } else { // birth, alive or dying
      int motionFlags, oldFlags;

      // Forward / back motion
      oldFlags = pLocalClient->GetMotionFlags();
      motionFlags = 0;

      strcpy( query.d_name, "accel+" );
      d_dinput.QueryAction( &query );
      if( query.d_state == ACTION_ON ) motionFlags |= MOTION_MOVEFORWARD;
      strcpy( query.d_name, "accel-" );
      d_dinput.QueryAction( &query );
      if( query.d_state == ACTION_ON ) motionFlags |= MOTION_MOVEBACKWARD;

      // Left / right turning
      strcpy( query.d_name, "turnLeft" );
      d_dinput.QueryAction( &query );
      if( query.d_state == ACTION_ON ) motionFlags |= MOTION_TURNLEFT;

      strcpy( query.d_name, "turnRight" );
      d_dinput.QueryAction( &query );
      if( query.d_state == ACTION_ON ) motionFlags |= MOTION_TURNRIGHT;

      // Weapons' fire
      strcpy( query.d_name, "firePrimary" );
      d_dinput.QueryAction( &query );
      if( query.d_state == ACTION_BEGUN ) motionFlags |= MOTION_FIRESTART;

      // BAD C++ - Now change this to work like events do...
      pLocalClient->SetMotionFlags( motionFlags );
    }
  }

  d_lastTime = currTime;

  // Now Render
  d_D2D.Clear();

  cameraMatrix.a30 = -pLocalClient->GetLocation()->a30;
  cameraMatrix.a31 = -pLocalClient->GetLocation()->a31;
  cameraMatrix.a32 = 300;  // Remember that's -(-300)

  for( i = 0; i < d_numPatches; i++ ) {
    // I had plans to have a bunch of gravity wells that all use
    // the same patch (scaled, of course) but I didn't have time
    d_patches[ i ].Update( &d_D2D, &cameraMatrix );
    d_patches[ i ].Render( &d_D2D );
  }

  for( itterClient = pLocalClient; itterClient(); ++itterClient ) {
    if( itterClient()->GetState() == STATE_ALIVE || itterClient()->GetState() == STATE_BIRTH ) {
      finalMatrix = *itterClient()->GetLocation() * cameraMatrix;
      itterClient()->Render( &d_D2D, &finalMatrix );
    }
  }

  for( itterMissile = d_clientThread.GetMissiles()->GetNext(); itterMissile(); ++itterMissile ) {
    finalMatrix = *itterMissile()->GetLocation() * cameraMatrix;
    itterMissile()->Render( &d_D2D, &finalMatrix );
  }

  // Render Lag-o-meter (only when logged in)
  if( pLocalClient->GetId() && d_D2D.GetWidth() > LAG_RECORDLENGTH * 2 && d_D2D.GetHeight() > 50 ) {
    int   startX, startY, pushLatency, *pLagRecord;
    char  lagString[ 25 ];  // just to be safe

    startX = ( d_D2D.GetWidth() / 2 ) - LAG_RECORDLENGTH;
    startY = d_D2D.GetHeight() - 11;
    pLagRecord = d_clientThread.GetLagRecord();
    pushLatency = d_clientThread.GetPushLatency() * TIME_PER_SNAPSHOT / 10;

    // Shows which frames are being interpolated between snapshots
    // and which are being extrapolated beyond the valid time.
    for( i = 0; i < LAG_RECORDLENGTH; i++ ) {
      if( pLagRecord[ i ] ) {
        if( pLagRecord[ i ] >= TIME_PER_SNAPSHOT ) {
          d_D2D.DrawLine( startX    , startY, startX    , startY - ( pLagRecord[ i ] / 10 ), 255, 255,   0, 255 );
          d_D2D.DrawLine( startX + 1, startY, startX + 1, startY - ( pLagRecord[ i ] / 10 ), 255, 255,   0, 255 );
        } else {
          d_D2D.DrawLine( startX    , startY, startX    , startY + ( pLagRecord[ i ] / 10 ),   0,   0, 255, 255 );
          d_D2D.DrawLine( startX + 1, startY, startX + 1, startY + ( pLagRecord[ i ] / 10 ),   0,   0, 255, 255 );
        }
      }
      startX += 2;
    }

    // Display optimal expected lag
    d_D2D.DrawLine( ( d_D2D.GetWidth() / 2 ) - LAG_RECORDLENGTH, ( startY - TIME_PER_SNAPSHOT / 10 ),
                    ( d_D2D.GetWidth() / 2 ) + LAG_RECORDLENGTH, ( startY - TIME_PER_SNAPSHOT / 10 ),
                    255, 0, 0, 255 );

    // Display average lag
    if( pushLatency > 50 ) pushLatency = 50;
    d_D2D.DrawLine( ( d_D2D.GetWidth() / 2 ) - LAG_RECORDLENGTH, ( startY - pushLatency ),
                    ( d_D2D.GetWidth() / 2 ) + LAG_RECORDLENGTH, ( startY - pushLatency ),
                    0, 255, 0, 255 );

    // Write the pushLatency number to the right of the graph.
    sprintf( lagString, "%d", d_clientThread.GetPushLatency() );
    d_D2D.FontWrite( ( d_D2D.GetWidth() / 2 ) + LAG_RECORDLENGTH + 10, startY - 25 - 4, lagString, 0, 0 );

    // Since all packets are received we don't need the other half of quake's lag-o-meter
    // that shows which packets are received, which are dropped and which are supressed.
  }

  // Render Console
  d_console.Render( "" );  // "" was d_dinput.d_keySequence (list of alphanumeric keys typed)

  // Add an FPS counter?  Nah, this isn't about rendering techniques...

  // display the rendered frame
  d_D2D.Render();

  return 0;
}
////////////////////////////////////////////////////////////////////////////////

LRESULT cNetworkClient::OnActivate( WPARAM wparam, LPARAM lparam ) {
  if( LOWORD( wparam ) == WA_INACTIVE ) d_isActive = 0;
  else {
    if( d_D2D.CheckSurfaceIntegrity() ) {
      cWindow::Destroy();
      return 1;
    }
    if( d_dinput.ReaquireAll() ) {
      cWindow::Destroy();
      return 1;
    }
    d_isActive = 1;
  }

  return cWindow::OnActivate( wparam, lparam );
}


LRESULT cNetworkClient::OnMove( WPARAM wparam, LPARAM lparam ) {
  if( d_isActive ) d_D2D.Move();

  return cWindow::OnMove( wparam, lparam );
}


LRESULT cNetworkClient::OnSize( WPARAM wparam, LPARAM lparam ) {
  if( wparam == SIZE_MAXHIDE || wparam == SIZE_MINIMIZED ) {
    d_isActive = 0;
    // start timer to read in messages from server
    // so that a huge queue doesn't build up.
  } else d_isActive = 1;

  if( d_isActive && d_isReady ) {
    RECT rect;

    d_isReady = 0;

    rect.left = 0;
    rect.top = 0;
    rect.right = LOWORD( lparam );
    rect.bottom = HIWORD( lparam );

#if defined( _DEBUG )
    d_D2D.Resize( &rect, 16, 0 );
#else
    d_D2D.Resize( &rect, 16, 1 );
#endif
    d_console.Resize( &rect );

    // stop timer, if any

    d_isReady = 1;
  }

  return cWindow::OnSize( wparam, lparam );
}


// Called once every 1/1000 of a second
LRESULT cNetworkClient::OnTimer( WPARAM wparam, LPARAM lparam ) {
  return cWindow::OnTimer( wparam, lparam );
}


LRESULT cNetworkClient::OnCommand( WPARAM wparam, LPARAM lparam ) {
  switch( LOWORD( wparam ) ) {
    case SC_MONITORPOWER:  return 1;  // prevent potential crashes
    default: break;
  }
  return cWindow::OnCommand( wparam, lparam );
}


LRESULT cNetworkClient::OnOther( unsigned int msg, WPARAM wparam, LPARAM lparam ) {
  switch( msg ) {
    case SM_GETHOST:  d_clientThread.OnGetHost( wparam, lparam );  break;
    case SM_ASYNC:  d_clientThread.OnAsync( wparam, lparam );  break;
  }

  return cWindow::OnOther( msg, wparam, lparam );
}

