#if !defined( __CCOMMCLIENTTHREAD_H )
#include "cCommClientThread.h"
#endif
////////////////////////////////////////////////////////////////////////////////

cCommClientThread::cCommClientThread() {
  d_startTime = 0;
  d_maxClients = 8;
  d_numClients = 0;
  d_pushLatency = 2;
  d_lastPacketReceived = 0;
  d_lastPacketSent = d_pushLatency;
  d_pLog = NULL;
  memset( d_lagRecord, 0, 100 * sizeof( int ) );
}


cCommClientThread::~cCommClientThread() {
  Shutdown();
}
////////////////////////////////////////////////////////////////////////////////

int cCommClientThread::Startup( HINSTANCE hInst, HINSTANCE hPrev, HWND hwnd, cLog *pLog ) {
  d_pLog = pLog;
  if( cComm::Init() ) return 1;
  if( cComm::Startup( MODE_CLIENT, hwnd ) ) return 2;
  d_startTime = GetTickCount();

  if( d_vehicles.LoadModel( NULL ) ) return 3;

  d_clients.SetVehicle( &d_vehicles );

  return 0;
}


int cCommClientThread::Shutdown() {
  tcLinkedList<cPlayer> *pClient;

#if defined( _DEBUG )
  // it can happen that Shutdown() is called before Startup().
  if( d_pLog ) d_pLog->Log( "Shutdown()\n" );
#endif

  cComm::Disconnect();

  while( ( pClient = d_clients.GetNext() ) ) delete pClient;
  d_vehicles.UnloadModel();

  return 0;
}
////////////////////////////////////////////////////////////////////////////////

int cCommClientThread::Update( DWORD deltaTime ) {
  tcLinkedList<cMissile>          *pMissile, *pNextMissile;
  tcLinkedListItterator<cPlayer>  itterClient;
  cVector3                        collide;
  int                             i;

  ParseQueuedMessages();

  // Stop now if the player is not logged into the game.
  if( !d_clients.GetSecurity() ) return 0;

  // Move.  Server handles collision detection.
  for( itterClient = &d_clients; itterClient(); ++itterClient ) {
    itterClient()->Update( deltaTime );
  }

  for( pMissile = d_missiles.GetNext(); pMissile; pMissile = pNextMissile ) {
    pNextMissile = pMissile->GetNext();
    if( pMissile->GetState() == STATE_DEAD ) {
      delete pMissile;
      continue;
    }
    pMissile->Update( deltaTime );
  }

  // Keep a record of how much lag we're suffering for the lag-o-meter
  for( i = 0; i < LAG_RECORDLENGTH - 1; i++ ) {
    d_lagRecord[ i ] = d_lagRecord[ i + 1 ];
  }
  d_lagRecord[ LAG_RECORDLENGTH - 1 ] = d_updateReadDelay;
  if( d_lagRecord[ LAG_RECORDLENGTH - 1 ] > 500 ) d_lagRecord[ LAG_RECORDLENGTH - 1 ] = 500;

  // Send message to server.
  if( d_updateSendDelay > TIME_PER_SNAPSHOT ) {
    if( d_clients.GetMotionFlags() != d_clients.GetOldMotionFlags() ) SendMoveNotice();
    else SendKeepAlive();

    d_lastPacketSent++;
    d_updateSendDelay -= TIME_PER_SNAPSHOT;
  }

  d_updateSendDelay += deltaTime;
  d_updateReadDelay += deltaTime;

  return 0;
}


int cCommClientThread::SendMoveNotice() {
  tcLinkedList<cMsgQueue> *pNewQueue;
  cMsgMoveNotice          *pNewMove;

  pNewMove = new cMsgMoveNotice;
  pNewQueue = new tcLinkedList<cMsgQueue>;
  if( !pNewQueue || !pNewMove ) {
    MessageBox( NULL, "Memory allocation failure!", "Error!", MB_APPLMODAL );
    // One may have been created...
    if( pNewQueue ) delete pNewQueue;
    if( pNewMove ) delete pNewMove;
    return 1;
  }

  pNewMove->d_type = MSG_MOVENOTICE;
  pNewMove->d_obj.d_id = d_clients.GetId();
  pNewMove->d_pos.Position( d_clients.GetLocation()->a30,
                            d_clients.GetLocation()->a31,
                            d_clients.GetLocation()->a32 );
  pNewMove->d_motionFlags = d_clients.GetMotionFlags();
  pNewMove->d_angle = d_clients.GetAngle();
  pNewMove->d_packetId = d_lastPacketSent + d_pushLatency;
  pNewQueue->SetBuffer( (void *)pNewMove, sizeof( cMsgMoveNotice ) );
  d_clients.AddOutMsgQueue( pNewQueue );
  WriteFromClientQueue( &d_clients );

  d_clients.SetOldMotionFlags( d_clients.GetMotionFlags() );
#if defined( _DEBUG )
  d_pLog->Log( "send %d m\n", d_lastPacketSent );
#endif

  return 0;
}


int cCommClientThread::SendKeepAlive() {
  tcLinkedList<cMsgQueue> *pNewQueue;
  cMsgKeepAlive           *pAlive;

  pAlive = new cMsgKeepAlive;
  pNewQueue = new tcLinkedList<cMsgQueue>;
  if( !pNewQueue || !pAlive ) {
    MessageBox( NULL, "Memory allocation failure!", "Error!", MB_APPLMODAL );
    // One may have been created...
    if( pNewQueue ) delete pNewQueue;
    if( pAlive ) delete pAlive;
    return 1;
  }

  pAlive->d_type = MSG_KEEPALIVE;
  pAlive->d_obj.d_type = OBJTYPE_PLAYER;
  pAlive->d_obj.d_id = d_clients.GetId();
  pAlive->d_packetId = d_lastPacketSent + d_pushLatency;
  pNewQueue->SetBuffer( (void *)pAlive, sizeof( cMsgKeepAlive ) );
  d_clients.AddOutMsgQueue( pNewQueue );
  WriteFromClientQueue( &d_clients );

#if defined( _DEBUG )
  d_pLog->Log( "send %d i\n", d_lastPacketSent );
#endif

  return 0;
}


// Bad C++ - This method would probably be more efficient as a tree
// of methods so as to reduce duplicate code & increase flexibility.
void cCommClientThread::ParseQueuedMessages() {
  tcLinkedList<cMsgQueue> *pQueue;

  while( d_clients.GetNextMsgIn() ) {
    pQueue = d_clients.GetNextMsgIn();
    if( !pQueue->GetSize() || pQueue->GetSize() != pQueue->GetTransmitted() ) break;

    // Find out what the server sent us.
    switch( ( (cMsg *)pQueue->GetBuffer() )->d_type ) {
      case MSG_LOGINREPLY:     ParseLoginReply( pQueue->GetBuffer() );     break;
      case MSG_CONNECTNOTICE:  ParseConnectNotice( pQueue->GetBuffer() );  break;
      case MSG_MOVENOTICE:     ParseMoveNotice( pQueue->GetBuffer() );     break;
      case MSG_SNAPSHOT:       ParseSnapshot( pQueue->GetBuffer() );       break;
      case MSG_SPAWNNOTICE:    ParseSpawnNotice( pQueue->GetBuffer() );    break;
      case MSG_COLLIDENOTICE:  ParseCollideNotice( pQueue->GetBuffer() );  break;
      case MSG_DEATHNOTICE:  // no longer sent by server
      case MSG_KEEPALIVE:  // is not sent by server
      default:  break;
    }

    // We're done with it, get rid of it.
    delete pQueue;
  }
}


int cCommClientThread::ParseLoginReply( void *pBuffer ) {
  cMsgLoginReply *pReply;

  pReply = (cMsgLoginReply *)pBuffer;

  switch( pReply->d_success ) {
    case LOGIN_OK:  // only reached in login ok.
      d_pLog->Log( "Login Approved\nPlayer id = %d\nLast packet sent = %d\n",
        pReply->d_id, pReply->d_lastPacketId );
      d_maxClients = pReply->d_maxClients;
      d_clients.SetId( pReply->d_id );
      d_numClients = 1;
      d_updateSendDelay = 0;
      d_updateReadDelay = 0;
      d_firstPacketReceived = pReply->d_lastPacketId;
      d_lastPacketReceived = pReply->d_lastPacketId;
      d_lastPacketSent = d_lastPacketReceived + 1;
      d_clients.SetSecurity( 1 );
      break;
    case LOGIN_BADFULL:      d_pLog->Log( "Login Denied - Server is full.\n" );                           break;
    case LOGIN_BADOUTDATED:  d_pLog->Log( "Login Denied - Client and server versions do not match.\n" );  break;
    case LOGIN_BADPASS:      d_pLog->Log( "Login Denied - Wrong password.\n" );                           break;
    case LOGIN_BADNAME:      d_pLog->Log( "Login Denied - Name already being used.\n" );                  break;
    case LOGIN_BADUNKNOWN:
    default:                 d_pLog->Log( "Login Denied - Reason unknown.\n" );                           break;
  }
  if( pReply->d_success != LOGIN_OK ) {
    d_pLog->Log( "Disconnect: pReply->d_success != LOGIN_OK\n" );
    Disconnect();
  }
  return 0;
}


int cCommClientThread::ParseConnectNotice( void *pBuffer ) {
  tcLinkedListItterator<cPlayer>  itterClient;
  cMsgConnectNotice               *pConnect;
  tcLinkedList<cPlayer>           *pClient;

  // There will never be a connectNotice for the local player
  pConnect = (cMsgConnectNotice *)pBuffer;

  for( itterClient = d_clients.GetNext(); itterClient(); ++itterClient ) {
    if( pConnect->d_obj.d_id == itterClient()->GetId() ) break;
  }
  // New client?
  if( !itterClient() && pConnect->d_connState != STATE_DISCONNECTING ) {
    pClient = new tcLinkedList<cPlayer>;
    if( !pClient ) {
      // crap...
      d_pLog->Error( "Not enough ram to create new player.\n" );
      Disconnect();
      return 2;
    }
    pClient->SetId( pConnect->d_obj.d_id );
    pClient->SetName( pConnect->d_name );
    pClient->SetVehicle( &d_vehicles );
    if( pConnect->d_connState == STATE_CONNECTED ) {
      pClient->SetState( pConnect->d_state );
      pClient->GetLocation()->a30 = pConnect->d_pos.d_x;
      pClient->GetLocation()->a31 = pConnect->d_pos.d_y;
      pClient->GetLocation()->a32 = pConnect->d_pos.d_z;
      pClient->SetMotionFlags( pConnect->d_motionFlags );
      pClient->SetAngle( pConnect->d_angle );
      d_pLog->Log( "Player %s (%d) is already playing.\n", pConnect->d_name, pConnect->d_obj.d_id );
    } else d_pLog->Log( "Player %s (%d) had entered.\n", pConnect->d_name, pConnect->d_obj.d_id );
    d_clients.Link( pClient );
    d_numClients++;
  } else if( itterClient() && pConnect->d_connState == STATE_DISCONNECTING ) {
    // print a message to the window?
    d_pLog->Log( "Player %s (%d) has left.\n", itterClient()->GetName(), itterClient()->GetId() );
    delete itterClient();
    d_numClients--;
  } else d_pLog->Error( "ConnectNotice error.\n" );
  return 0;
}


int cCommClientThread::ParseMoveNotice( void *pBuffer ) {
  cVector3                        predict, pos;
  tcLinkedListItterator<cPlayer>  itterClient;
  cMsgMoveNotice                  *pMove;
  float                           angle;

  // MoveNotice should not be received by client but just in case
  // the client gets the wrong size for the snapshot be ready...

  pMove = (cMsgMoveNotice *)pBuffer;
  for( itterClient = &d_clients; itterClient(); ++itterClient ) {
    if( pMove->d_obj.d_id == itterClient()->GetId() ) break;
  }
  if( !itterClient() ) return 1;

  if( pMove->d_motionFlags & MOTION_TURNMASK != MOTION_TURNMASK ) {
    if( pMove->d_motionFlags & MOTION_TURNLEFT ) {
      angle = itterClient()->GetVehicle()->GetMaxTurnVel() * ( d_pushLatency * TIME_PER_SNAPSHOT / 1000.0f );
    } else {  // turn right
      angle = itterClient()->GetVehicle()->GetMaxTurnVel() * -( d_pushLatency * TIME_PER_SNAPSHOT / 1000.0f );
    }
    pos = itterClient()->GetVelocity();
    pos.RotateAboutLine( cVector3( 0, 0, -1 ), angle );  // +z axis is INTO screen.  Right hand rule.
    predict = pMove->d_pos +
              ( itterClient()->GetVelocity() * ( d_pushLatency * TIME_PER_SNAPSHOT / 2000.0f ) ) +
              ( pos * ( d_pushLatency * TIME_PER_SNAPSHOT / 2000.0f ) );
  } else if( pMove->d_motionFlags & MOTION_MOVEMASK != MOTION_MOVEMASK ) {
    predict = pMove->d_pos + ( itterClient()->GetVelocity() * ( d_pushLatency * TIME_PER_SNAPSHOT / 1000.0f ) );
  } else predict = pMove->d_pos;

  if( itterClient()->GetIdleTime() >= 1000 ) {
    itterClient()->GetLocation()->a30 = predict.d_x;
    itterClient()->GetLocation()->a31 = predict.d_y;
    itterClient()->GetLocation()->a32 = predict.d_z;
    itterClient()->SetMotionFlags( pMove->d_motionFlags );
    itterClient()->SetAngle( pMove->d_angle );
  } else {
    pos.Position( itterClient()->GetLocation()->a30,
                  itterClient()->GetLocation()->a31,
                  itterClient()->GetLocation()->a32 );
    pos -= predict;
    if( pos.Length() >= 5.0f ) {
      itterClient()->GetLocation()->a30 = predict.d_x;
      itterClient()->GetLocation()->a31 = predict.d_y;
      itterClient()->GetLocation()->a32 = predict.d_z;
    }
    itterClient()->SetAngle( pMove->d_angle );
  }
  itterClient()->SetIdleTime( 0 );
  return 0;
}


int cCommClientThread::ParseSnapshot( void *pBuffer ) {
  cMsgMoveNotice  *pSnapMove;
  cMsgSnapshot    *pSnap;
  int             i;

  pSnap = (cMsgSnapshot *)pBuffer;
  pSnapMove = (cMsgMoveNotice *)( (char *)pBuffer + sizeof( cMsgSnapshot ) );

  // determine the current pushLatency
  d_lastPacketReceived = pSnap->d_packetId;
  if( pSnap->d_lastPacketReceived != d_firstPacketReceived ) {
    d_pushLatency = pSnap->d_packetId - pSnap->d_lastPacketReceived;
    if( d_pushLatency < 1 ) d_pushLatency = 1;
  }
#if defined( _DEBUG )
  d_pLog->Log( "snap %d %d - %d\n", pSnap->d_packetId, pSnap->d_lastPacketReceived, d_pushLatency );
#endif
  d_updateReadDelay = 0;

  // now process the move notices
  for( i = 0; i < pSnap->d_numClients; i++ ) {
    ParseMoveNotice( (void *)pSnapMove );
    pSnapMove++;
  }

  return 0;
}


int cCommClientThread::ParseSpawnNotice( void *pBuffer ) {
  tcLinkedList<cMissile>          *pNewMissile;
  tcLinkedListItterator<cPlayer>  itterClient;
  cMsgSpawnNotice                 *pSpawn;

  pSpawn = (cMsgSpawnNotice *)pBuffer;
  if( pSpawn->d_obj.d_type == OBJTYPE_PLAYER ) {
    for( itterClient = &d_clients; itterClient(); ++itterClient ) {
      if( pSpawn->d_obj.d_id == itterClient()->GetId() ) break;
    }
    if( !itterClient() ) return 1;
    itterClient()->SetState( STATE_BIRTH );
    itterClient()->GetLocation()->a30 = pSpawn->d_pos.d_x;
    itterClient()->GetLocation()->a31 = pSpawn->d_pos.d_y;
    itterClient()->GetLocation()->a32 = pSpawn->d_pos.d_z;
    itterClient()->SetAngle( pSpawn->d_angle );
    d_pLog->Log( "Player %s (%d) spawned.\n", itterClient()->GetName(), itterClient()->GetId() );
  } else if( pSpawn->d_obj.d_type = OBJTYPE_MISSILE ) {
    // Create a new missile.
    pNewMissile = new tcLinkedList<cMissile>;
    if( !pNewMissile ) {
      // Crap...
      d_pLog->Error( "Not enough ram to create new missile.\n" );
      Disconnect();
      return 2;
    }

    for( itterClient = &d_clients; itterClient(); ++itterClient ) {
      if( pSpawn->d_creator.d_id == itterClient()->GetId() ) break;
    }
    if( !itterClient() ) return 1;
    // Assign it the data.
    pNewMissile->Fire( pSpawn->d_pos, 50.0f, pSpawn->d_angle );
    pNewMissile->SetCreator( pSpawn->d_creator.d_id );
    pNewMissile->SetId( pSpawn->d_obj.d_id );
    d_missiles.Link( pNewMissile );
    d_pLog->Log( "Player %s (%d) fired a missile.\n", itterClient()->GetName(), itterClient()->GetId() );
  }
  return 0;
}


int cCommClientThread::ParseCollideNotice( void *pBuffer ) {
  tcLinkedListItterator<cPlayer>  itterClient, itterOther;
  tcLinkedListItterator<cMissile> itterMissile;
  cMsgCollideNotice               *pColl;

  pColl = (cMsgCollideNotice *)pBuffer;
  if( pColl->d_colliders[ 0 ].d_type == OBJTYPE_PLAYER ) {
    for( itterClient = &d_clients; itterClient(); ++itterClient ) {
      if( pColl->d_colliders[ 0 ].d_id == itterClient()->GetId() ) break;
    }
    if( !itterClient() ) return 1;
    if( itterClient() == &d_clients ) d_pLog->Log( "You have collided with " );
    else d_pLog->Log( "%s has collided with ", itterClient()->GetName() );
    // Apply damage appropriate to collision type.
    // For this game we consider any collision to be fatal
    itterClient()->SetState( STATE_DYING );
  }

  if( pColl->d_colliders[ 1 ].d_type == OBJTYPE_PLAYER ) {
    for( itterOther = &d_clients; itterOther(); ++itterOther ) {
      if( pColl->d_colliders[ 1 ].d_id == itterOther()->GetId() ) break;
    }
    if( !itterOther() ) return 1;
    if( itterOther() == &d_clients ) d_pLog->Log( "you.\n" );
    else d_pLog->Log( "%s.\n", itterOther()->GetName() );
    // Apply damage appropriate to collision type.
    // For this game we consider any collision to be fatal
    itterOther()->SetState( STATE_DYING );
  } else if( pColl->d_colliders[ 1 ].d_type = OBJTYPE_MISSILE ) {
    for( itterMissile = d_missiles.GetNext(); itterMissile(); ++itterMissile ) {
      if( pColl->d_colliders[ 1 ].d_id == itterMissile()->GetId() ) break;
    }
    if( !itterMissile() ) return 1;
    for( itterOther = &d_clients; itterOther(); ++itterOther ) {
      if( itterMissile()->GetCreator() == itterOther()->GetId() ) break;
    }
    if( !itterOther() ) return 1;
    if( itterClient() != itterOther() ) {
      d_pLog->Log( "one of %s's missiles.\n", itterOther()->GetName() );
    } else d_pLog->Log( "one of your own missiles!\n" );
    // set the missile to blow up.
    itterMissile()->SetState( STATE_DYING );
  }

  return 0;
}
////////////////////////////////////////////////////////////////////////////////

int cCommClientThread::OnGetHost( WPARAM wparam, LPARAM lparam ) {
  LRESULT returnVal;

#if defined( _DEBUG )
  d_pLog->Log( "OnGetHost()\n" );
#endif

  returnVal = cComm::OnGetHost( wparam, lparam );
  if( returnVal ) {
    d_pLog->Error( "Difficulties encountered resolving host address.\n" );
    return 0;
  }

#if !defined( UDP_PROTOCOL )
  // Attempt to connect.  Whatever the results, SM_ASYNC
  // will be sent to the window with the FD_CONNECT flag.
  if( AttemptConnect() == WSAEINVAL ) {
    d_pLog->Error( "Connection attempt failed!\n" );
  }
#endif

  return 0;
}


LRESULT cCommClientThread::OnAsync( WPARAM wparam, LPARAM lparam ) {
  int                             event, errmsg;
  tcLinkedListItterator<cPlayer>  itter;

  event = WSAGETSELECTEVENT( lparam );
  errmsg = WSAGETSELECTERROR( lparam );
  switch( event ) {
    case FD_CONNECT:
      // Every time WSAAsyncSelect is called it sends messages to SM_ASYNC where it sends
      // the error code from WSAAsyncSelect, if any.  This means we may get repeated calls
      // to FD_CONNECT and FD_WRITE we don't want, under certain circumstances.
      d_pLog->Log( "FD_CONNECT\n" );

      if( errmsg ) {
        d_pLog->Error( "Error (%d).\n", errmsg );
        // Depending on the error type we might just want to try connecting again.
        break;
      }
      if( (SOCKET)wparam != GetSocket() || d_clients.GetSocket() != INVALID_SOCKET ) break;

      tcLinkedList<cMsgQueue> *pNewQueue;
      cMsgLoginRequest        *pNewMsg;

      d_startTime = GetTickCount();
      d_clients.SetSocket( (SOCKET)wparam );
      cComm::Connected( 0 );

      // Request login
      pNewMsg = new cMsgLoginRequest;
      pNewQueue = new tcLinkedList<cMsgQueue>;
      if( !pNewMsg || !pNewQueue ) {
        if( pNewMsg ) delete pNewMsg;
        if( pNewQueue ) delete pNewQueue;
        d_pLog->Error( "No memory left to complete login reply.\n" );
        Disconnect();
        return 1;
      }

      d_clients.SetName( d_name );
      pNewMsg->d_type = MSG_LOGINREQUEST;
      pNewMsg->d_version[ 0 ] = NC_VERSIONMAJOR;
      pNewMsg->d_version[ 1 ] = NC_VERSIONMINOR;
      strcpy( pNewMsg->d_name, d_name );
      strcpy( pNewMsg->d_pass, d_pass );
      pNewQueue->SetBuffer( (void *)pNewMsg, sizeof( cMsgLoginRequest ) );
      d_clients.AddOutMsgQueue( pNewQueue );
      WriteFromClientQueue( &d_clients );
      d_pLog->Log( "Sending login request\n" );
      break;
    case FD_READ:  // Incomming data to be read
      ReadIntoClientQueue( &d_clients );
      break;
    case FD_WRITE:  // Permission granted to send data
      WriteFromClientQueue( &d_clients );
      break;
    case FD_CLOSE:  // Disconnecting
      // At this point we probably can't write to the socket because
      // of shutdown( socket, SW_READ )...
      d_pLog->Log( "FD_CLOSE\nIf you see this it is because you have been disconnected by the server.\n" );
      Shutdown();
      break;
  }

  return cComm::OnAsync( wparam, lparam );
}


long cCommClientThread::DetermineSize( int type ) {
  switch( (eMsgType)type ) {
    case MSG_DEFAULT:  return sizeof( cMsg );
    case MSG_LOGINREQUEST:  return sizeof( cMsgLoginRequest );
    case MSG_LOGINREPLY:  return sizeof( cMsgLoginReply );
    case MSG_CONNECTNOTICE:  return sizeof( cMsgConnectNotice );
    case MSG_SNAPSHOT:  return sizeof( cMsgSnapshot ) + d_numClients * sizeof( cMsgMoveNotice );
    case MSG_MOVENOTICE:  return sizeof( cMsgMoveNotice );
    case MSG_SPAWNNOTICE:  return sizeof( cMsgSpawnNotice );
    case MSG_DEATHNOTICE:  return sizeof( cMsgDeathNotice );
    case MSG_COLLIDENOTICE:  return sizeof( cMsgCollideNotice );
    case MSG_KEEPALIVE:  return sizeof( cMsgKeepAlive );
    default:  return sizeof( cMsg );
  }
}

