/* Project: ARF Windows API socket server/client
 * File   : cComm.cpp
 * Created: 27/07/98 Dan Royer
 * Note   :
 */
#ifndef __CCOMM_H
#include "cComm.h"
#endif
#if !defined( __CCLIENT_H )
#include "cClient.h"
#endif
#include <stdio.h>
////////////////////////////////////////////////////////////////////////////////
cComm::cComm() {
  Init();
}


cComm::~cComm() {
  Shutdown();
}


int cComm::Init() {
  d_winsockVersion = MAKEWORD( WINSOCK_VERSIONMAJOR, WINSOCK_VERSIONMINOR );
  d_state = STATE_DISCONNECTED;
  d_serverPort = DEFAULT_SERVERPORT;
  strcpy( d_sessionName, "[ Session name will appear here ]" );
  strcpy( d_serverName, "[ Server name will appear here ]" );
  d_IP[ 0 ] = NULL;
  d_socket = INVALID_SOCKET;
  d_hwnd = NULL;

  return 0;
}


int cComm::Startup( int runFlag, HWND hwnd ) {
  char  buffer[ 80 ];
  int   error;

  d_hwnd = hwnd;
  d_runFlag = runFlag;  // server or client?
  d_socketFlags = FD_READ | FD_WRITE | FD_CONNECT | FD_CLOSE;
  if( d_runFlag ) {
#if !defined( UDP_PROTOCOL )
    d_socketFlags |= FD_ACCEPT;
#endif
    d_socketFlags |= FD_OOB;
  }

  // Winsock startup
  error = WSAStartup( d_winsockVersion, &d_winsockData );
  if( error == SOCKET_ERROR ) {
    if( error == WSAVERNOTSUPPORTED ) {
      sprintf( buffer, "WSAStartup error.\nRequested Winsock v%d.%d, found v%d.%d.", 
        WINSOCK_VERSIONMAJOR, WINSOCK_VERSIONMINOR, 
        LOBYTE( d_winsockData.wVersion ), HIBYTE( d_winsockData.wVersion ) );
      WSACleanup();
    } else sprintf( buffer, "WSAStartup error (%d)", WSAGetLastError() );
    MessageBox( NULL, buffer, "cComm::Startup()", MB_OK );  // Bad C++, what if DX window is open?
    return 1;
  }

  gethostname( d_serverName, sizeof( d_serverName ) );
  d_serverPort = DEFAULT_SERVERPORT;

  return 0;
}


int cComm::Shutdown() {
  Disconnect();  // just in case
  WSACleanup();
  return 0;
}


HANDLE cComm::GetHost( char *address, int port ) {
  if( d_state != STATE_DISCONNECTED ) return NULL;

  d_state = STATE_CONNECTING;

  if( port ) d_serverPort = port;
  if( address ) strcpy( d_serverName, address );

  d_address.sin_addr.s_addr = inet_addr( d_serverName );  // htonl could be used instead of inet_addr
  d_address.sin_port = htons( d_serverPort );
  d_address.sin_family = AF_INET;

  if( d_address.sin_addr.s_addr == INADDR_NONE ) {
    d_hostHandle = WSAAsyncGetHostByName( d_hwnd, SM_GETHOST, d_serverName, d_hostentbuf, MAXGETHOSTSTRUCT );
  } else {
    d_hostHandle = WSAAsyncGetHostByAddr( d_hwnd, SM_GETHOST, (const char *)&d_address.sin_addr.s_addr, sizeof( IN_ADDR ), AF_INET, d_hostentbuf, MAXGETHOSTSTRUCT );
  }

  return d_hostHandle;
}


int cComm::OnGetHost( WPARAM wparam, LPARAM lparam ) {
  if( (HANDLE)wparam != d_hostHandle ) return 1;
  if( WSAGETASYNCERROR( lparam ) ) {  // upper 16 bits of (DWORD?)
    Disconnect();
    return 2;
  }

  if( CreateSocket() == INVALID_SOCKET ) {
    Disconnect();
    return 3;
  }

  if( ChangeWindows( d_hwnd ) == SOCKET_ERROR ) {
    Disconnect();
    return 4;
  }

  if( AddressHost() ) {
    Disconnect();
    return 5;
  }

  return 0;
}


SOCKET cComm::CreateSocket() {
  if( d_socket != INVALID_SOCKET ) return d_socket;  // already exists

#if defined( UDP_PROTOCOL )
  d_socket = socket( AF_INET, SOCK_DGRAM, 0 );
#else
  d_socket = socket( AF_INET, SOCK_STREAM, 0 );
#endif

  return d_socket;
}


int cComm::ChangeWindows( HWND hwnd ) {
  d_hwnd = hwnd;

  return WSAAsyncSelect( d_socket, d_hwnd, SM_ASYNC, d_socketFlags );
}


int cComm::AddressHost() {
  LPHOSTENT lphostent;

  lphostent = (LPHOSTENT)d_hostentbuf;
  // Bad C++ - check lphostent = NULL?
  strcpy( d_IP, inet_ntoa( *(LPIN_ADDR)lphostent->h_addr_list[ 0 ] ) );

  if( !d_runFlag ) {
    memcpy( &d_address.sin_addr, (LPIN_ADDR)lphostent->h_addr_list[ 0 ], sizeof( IN_ADDR ) );
  } else {  // server code
    d_address.sin_addr.s_addr = INADDR_ANY;
    if( bind( d_socket, (LPSOCKADDR)&d_address, sizeof( SOCKADDR ) ) == SOCKET_ERROR ) {
      closesocket( d_socket );
      return 1;
    }
#if !defined( UDP_PROTOCOL )
    if( listen( d_socket, SOMAXCONN ) == SOCKET_ERROR ) {
      closesocket( d_socket );
      return 2;
    }
#endif
  }
  return 0;
}


int cComm::AttemptConnect() {
  return connect( d_socket, (LPSOCKADDR)&d_address, sizeof( SOCKADDR_IN ) );
}


int cComm::Connected( int errmsg ) {
  // Chance to quit if something went wrong, print appropriate message here.
  d_state = STATE_CONNECTED;

  return 0;
}


int cComm::Disconnect() {
  switch( d_state ) {
    case STATE_CONNECTED:
      d_state = STATE_DISCONNECTING;
      return shutdown( d_socket, SD_SEND );
    case STATE_CONNECTING:
      // there is no pending data
      closesocket( d_socket );
      d_socket = INVALID_SOCKET;
      d_state = STATE_DISCONNECTED;
      break;
    case STATE_DISCONNECTING:
      // read in any pending data on the socket
      closesocket( d_socket );
      d_socket = INVALID_SOCKET;
      d_state = STATE_DISCONNECTED;
      break;
    default: break;
  }

  return 0;
}


int cComm::OnAsync( WPARAM wparam, LPARAM lparam ) {
  return 1;
}


int cComm::ReadIntoClientQueue( cClient *pClient ) {
  long                    error, lasterror, size;
  char                    *pBuffer;
  tcLinkedList<cMsgQueue> *pQueue;
  int                     type;
#if defined( UDP_PROTOCOL )
  SOCKADDR                from;
  int                     fromLen;
#endif

  pQueue = NULL;

  // Can only be receiving one data block at a time per client
  if( !pClient->GetReceiving() ) {
#if defined( UDP_PROTOCOL )
    error = recvfrom( pClient->GetSocket(), (char *)&type, sizeof( int ), &from, &fromLen );
#else
    error = recv( pClient->GetSocket(), (char *)&type, sizeof( int ), 0 );
#endif
    if( error == SOCKET_ERROR ) {
      if( WSAGetLastError() == WSAEWOULDBLOCK ) return 2;
      else return WSAGetLastError();
    }

    size = DetermineSize( type );

    pBuffer = new char[ size ];  // Allocate a buffer of needed size
    if( !pBuffer ) return 3;  // uh oh
    pQueue = new tcLinkedList<cMsgQueue>;
    if( !pQueue ) {
      delete pBuffer;
      return 4;
    }
    pQueue->SetBuffer( pBuffer, size, 1 );  // 1 - was allocated as an array
    ( (cMsg *)pBuffer )->d_type = type;

    pQueue->AddTransmitted( error );  // we've already read in pMsgId, so compensate
    pClient->AddInMsgQueue( pQueue );
    size -= error;
    if( !size ) return 0;

    pClient->SetReceiving( 1 );
    pBuffer += error;
  } else {
    tcLinkedListItterator<cMsgQueue> itter;

    for( itter = pClient->GetNextMsgIn(); itter(); ++itter ) {
      if( itter()->GetSize() != itter()->GetTransmitted() ) break;
    }
    if( !itter() ) return 5;  // big problem

    pQueue = itter();
    pBuffer = (char *)pQueue->GetBuffer() + pQueue->GetTransmitted();
    size = pQueue->GetSize() - pQueue->GetTransmitted();
  }

#if defined( UDP_PROTOCOL )
  error = recvfrom( pClient->GetSocket(), pBuffer, sizeof( int ), 0, &from, &fromLen );
#else
  error = recv( pClient->GetSocket(), pBuffer, size, 0 );
#endif
  if( error == SOCKET_ERROR ) {
    lasterror = WSAGetLastError();
    if( lasterror != WSAEWOULDBLOCK ) {
      return WSAGetLastError();
    }
    if( lasterror != WSAEMSGSIZE ) error = 0;
    // in the event that it IS WSAEpMsgSIZE we succesfully read in the data
    // so don't go apeshit.
  }

  if( !( size - error ) ) {
    pClient->SetReceiving( 0 );
    pQueue->AddTransmitted( error );
    return 0;
  }

  pQueue->AddTransmitted( error );
  return 1;
}


int cComm::WriteFromClientQueue( cClient *pClient ) {
  int                     error, length;
  tcLinkedList<cMsgQueue> *pMsg;
  char                    *ptr;
#if defined( UDP_PROTOCOL )
  int                     toLen;
  SOCKADDR                to;
#endif

  pMsg = pClient->GetNextMsgOut();
  if( !pMsg ) return 3;  // shouldn't happen...

  ptr = (char *)pMsg->GetBuffer() + pMsg->GetTransmitted();
  length = pMsg->GetSize() - pMsg->GetTransmitted();

#if defined( UDP_PROTOCOL )
  // A serious modification would have to be made to client for this
  // to work server-side and still work client-side (to and toLen
  // need to be initialized) and I'm not feeling up to it right now.
  error = sendto( pClient->GetSocket(), ptr, length, 0, pClient->GetAddress(), sizeof( SOCKADDR ) );
#else
  error = send( pClient->GetSocket(), ptr, length, 0 );
#endif
  if( error == SOCKET_ERROR ) {
    if( WSAGetLastError() == WSAEWOULDBLOCK ) error = 0;
    else return WSAGetLastError();
  }

  ptr += error;
  length -= error;
  pMsg->AddTransmitted( error );

  if( !length ) {
    delete pMsg;

    return 0;
  }

  return 0;
}


int cComm::Debug() {
  FILE *fDebug;
  fDebug = fopen( "debug.txt", "at" );
  fprintf( fDebug, "cComm --------------------\n" );
  fprintf( fDebug, "d_serverPort =%d\n", d_serverPort );
  fprintf( fDebug, "d_serverName =%s\n", d_serverName );
  fprintf( fDebug, "d_socketFlags=%d\n", d_socketFlags );
  fclose( fDebug );

  return 0;
}
////////////////////////////////////////////////////////////////////////////////
// EOF /////////////////////////////////////////////////////////////////////////
