// Wnd.cpp

#include "stdafx.h"
#include "wnds.h"

/////////////////////////////////////////////////////////////////////////////
// Handle Map

static CWndMap WndMap;

/////////////////////////////////////////////////////////////////////////////
// Message routing

LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch(uMsg)
	{

	// Windows
	case WM_CREATE:
		{
			if(IsWindow(hWnd))
			{
				// Window instance passed in CreateWindow()
				CWnd* pWnd = NULL;

				// Get a CREATESTRUCT
				LPCREATESTRUCT pCreateStruct = reinterpret_cast<LPCREATESTRUCT>(lParam);

				// Extract the pointer to the window
				if(pCreateStruct)
					pWnd = static_cast<CWnd*>(pCreateStruct->lpCreateParams);

				// Attach C++ class to windows handle
				if(pWnd)
				{
					// Created
					if(pWnd->Msg(hWnd, uMsg, wParam, lParam))
					{
						// Set the window handle
						pWnd->m_hWnd = hWnd;

						// And mapped...
						WndMap[hWnd] = pWnd;

						// Diagnostics
						DumpHandleMap();

						// Success
						return FALSE;
					}
				}
			}
		}

		break;


	// Dialogs
	case WM_INITDIALOG:
		{
			if(IsWindow(hWnd))
			{
				// Window instance passed in CreateDialog()
				CDialog* pDlg = reinterpret_cast<CDialog*>(lParam);

				// Attach C++ class to windows handle
				if(pDlg)
				{
					// Set the window handle
					pDlg->m_hWnd = hWnd;

					// Map it...
					WndMap[hWnd] = pDlg;

					// Diagnostics
					DumpHandleMap();

					// Created
					return pDlg->Msg(hWnd, uMsg, wParam, lParam);
				}
			}
		}

		break;


	// Clean up...
	case WM_DESTROY:
		{
			if(IsWindow(hWnd))
			{
				// Look for this window in the handle map
				if(WndMap.find(hWnd) != WndMap.end())
				{
					// Get a pointer to window
					CWnd* pWnd = WndMap[hWnd];

					if(pWnd)
					{
						//  Let window handle WM_DESTROY
						pWnd->Msg(hWnd, uMsg, wParam, lParam);

						// Free the C++ object
						delete pWnd;

						// Erase the handle mapping
						WndMap.erase(hWnd);

						// Diagnostics
						DumpHandleMap();
					}
				}
			}
		}

		break;


	default:
		{
			// Look for this window in the handle map
			if(WndMap.find(hWnd) != WndMap.end())
			{
				// Get a pointer to window
				CWnd* pWnd = WndMap[hWnd];

				// Route the message to the correct window
				if(pWnd)
					return pWnd->Msg(hWnd, uMsg, wParam, lParam);
			}
		}
	}

	return (DefWindowProc(hWnd, uMsg, wParam, lParam));
}

/////////////////////////////////////////////////////////////////////////////
// Window creation

template <class T> T* CreateWnd(HINSTANCE hInstance, LPCTSTR szClass, 
								DWORD dwStyles, LPCTSTR szTitle /* = NULL */, HWND hParent /* = NULL */)
{
	WNDCLASSEX wcex;
	ZeroMemory(&wcex, sizeof (WNDCLASSEX));
	wcex.cbSize        = sizeof(WNDCLASSEX);
    wcex.style         = 0;
    wcex.lpfnWndProc   = WndProc;
    wcex.cbClsExtra    = 0;
    wcex.cbWndExtra    = 0;
    wcex.hInstance     = hInstance;
    wcex.hIcon         = 0;
    wcex.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground = HBRUSH(GetSysColorBrush(COLOR_BTNFACE));
    wcex.lpszMenuName  = NULL;
    wcex.lpszClassName = szClass;
    wcex.hIconSm       = NULL;

    // Register the window class
    RegisterClassEx(&wcex);

	// New window
	T* pWnd = new T();

	HWND hWnd = CreateWindow(
            szClass,					// Class of the window to create
            szTitle,					// Window caption text
            dwStyles,					// Window styles
            CW_USEDEFAULT, 0,
			CW_USEDEFAULT, 0,			
            hParent,					// Parent window handle
            (HMENU)0,					// Child window ID
            hInstance,					// Instance of module owning window
            pWnd);						// Pass a pointer to the derived class

	if(IsWindow(hWnd))
	{
		ShowWindow(hWnd, SW_SHOW);
		UpdateWindow(hWnd);

		// Find the item, perform sanity check
		if(WndMap.find(hWnd) != WndMap.end())
			pWnd = static_cast<T*>(WndMap[hWnd]);

	} else
	{
		// Bad error, eject!
		delete pWnd, pWnd = NULL;
	}

	return pWnd;
}

/////////////////////////////////////////////////////////////////////////////
// Dialog creation

template <class T> T* CreateDlg(HINSTANCE hInstance, LPCTSTR lpTemplate, HWND hParent /*= NULL*/)
{
	// New dialog
	T* pDlg = new T();

	// Sanity
	if(!pDlg)
		return NULL;

	// Create the dialog
	HWND hDlg = CreateDialogParam(
		hInstance,						// Instance of module owning window
		lpTemplate,						// Dialog template
		hParent,						// Parent window handle
		DLGPROC(WndProc),				// Our global window procedure
		LPARAM(pDlg));					// Pointer to the dialog

	if(IsWindow(hDlg))
	{
		ShowWindow(hDlg, SW_SHOW);
		UpdateWindow(hDlg);

		// Find the item, perform sanity check
		if(WndMap.find(hDlg) != WndMap.end())
			pDlg = static_cast<T*>(WndMap[hDlg]);

	} else
	{
		// Bad error, eject!
		delete pDlg, pDlg = NULL;
	}

	return pDlg;
}

/////////////////////////////////////////////////////////////////////////////
// Diagnostics

void DumpHandleMap()
{
	// Work through every item
	iWnd Begin = WndMap.begin();
	iWnd End = WndMap.end();

	for(iWnd Wnd = Begin; Wnd != End; Wnd++)
	{
		// Get a pointer to the item
		CWnd* pWnd = Wnd->second;

		// Add it if it is there...
		if(pWnd)
		{
			// Get window handle
			HWND hWnd = pWnd->m_hWnd;

			if(IsWindow(hWnd))
			{
				// Get window text
				ALLOCBUFFER(szText, 1024)
				GetWindowText(hWnd, szText, 1024);

				// Format report
				ALLOCBUFFER(szDebug, 1024 + 128)
				_stprintf(szDebug, _T("Window Handle->0x%08x Text->\'%s\'\n"), hWnd, szText);

				// Display
				OutputDebugString(szDebug);

			} else
			{
				// Display
				OutputDebugString(_T("WARNING: Mapped window does not exist!\n"));
			}
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
// Derived window classes

DECLARE_WINDOW_CLASS(CPicture)

